// public/js/video-gallery.js

const API_URL = `${window.location.origin}/api/videos?limit=all`;

document.addEventListener("DOMContentLoaded", async () => {
    const wrap = document.getElementById("videoGalleryWrap");
    if (!wrap) return;

    try {
        const res = await fetch(API_URL);
        if (!res.ok) throw new Error("API error: " + res.status);

        const videos = await res.json();
        wrap.innerHTML = "";

        if (!Array.isArray(videos) || videos.length === 0) {
            wrap.innerHTML = `<div class="col-12"><p>No videos found.</p></div>`;
            return;
        }

        videos.forEach((v, i) => {
            const delay = (0.1 * (i + 1)).toFixed(1);

            // Get video URL
            const rawVideo = v.video || v.tiktok_url || "";
            if (!rawVideo) {
                console.warn(`Video ${v.id} has no video URL, skipping`);
                return;
            }
            
            const videoUrl = toFullUrl(rawVideo);

            const isFileVideo = isLikelyVideoFileUrl(videoUrl);
            const isTikTok = isTikTokUrl(videoUrl);
            const isYouTube = isYouTubeUrl(videoUrl);

            // Create video element for uploaded files, thumbnail for external videos
            let content;
            if (isFileVideo) {
                // Show actual video player for uploaded videos - autoplay, muted, loop, no controls
                content = `
          <div class="video-gallery-image wow fadeInUp" data-wow-delay="${delay}s">
            <figure style="margin: 0;">
              <video 
                autoplay 
                muted 
                loop 
                playsinline
                style="width: 100%; height: 300px; object-fit: cover; background: #000; cursor: pointer;"
                onclick="this.muted = !this.muted;">
                <source src="${escapeAttribute(videoUrl)}" type="video/mp4">
                Your browser does not support the video tag.
              </video>
            </figure>
          </div>`;
            } else if (isYouTube) {
                // YouTube videos - extract video ID and show thumbnail
                const youtubeId = extractYouTubeId(videoUrl);
                const thumbUrl = youtubeId ? `https://img.youtube.com/vi/${youtubeId}/maxresdefault.jpg` : "/images/video-gallery-img-1.jpg";
                content = `
          <div class="video-gallery-image wow fadeInUp" data-wow-delay="${delay}s" data-cursor-text="Play">
            <a href="${escapeAttribute(videoUrl)}" class="popup-video">
              <figure>
                <img src="${escapeAttribute(thumbUrl)}" alt="${escapeHtml(v.title || "Video")}" style="width: 100%; height: 300px; object-fit: cover;">
              </figure>
            </a>
          </div>`;
            } else {
                // TikTok and other external videos
                content = `
          <div class="video-gallery-image wow fadeInUp" data-wow-delay="${delay}s" data-cursor-text="Play">
            <a href="${escapeAttribute(videoUrl)}" target="_blank" rel="noopener">
              <figure>
                <img src="/images/video-gallery-img-1.jpg" alt="${escapeHtml(v.title || "Video")}" style="width: 100%; height: 300px; object-fit: cover;">
              </figure>
            </a>
          </div>`;
            }

            wrap.insertAdjacentHTML(
                "beforeend",
                `<div class="col-lg-4 col-md-6">${content}</div>`
            );
        });

        // Re-init WOW animations (because content injected)
        if (typeof WOW === "function") {
            new WOW().init();
        }

        // Re-bind Magnific Popup for newly injected links
        if (
            window.jQuery &&
            window.jQuery.fn &&
            window.jQuery.fn.magnificPopup
        ) {
            window.jQuery(wrap).find(".popup-video").magnificPopup({
                type: "iframe",
                mainClass: "mfp-fade",
                removalDelay: 160,
                preloader: false,
                fixedContentPos: true,
            });
        }
    } catch (err) {
        console.error("Video gallery load failed:", err);
        wrap.innerHTML = `<div class="col-12"><p>Video gallery failed to load.</p></div>`;
    }
});

function toFullUrl(value) {
    if (!value) return "";

    // Already full URL (youtube or http link)
    if (
        String(value).startsWith("http://") ||
        String(value).startsWith("https://")
    ) {
        return value;
    }

    const s = String(value);

    // Already a public storage URL path
    if (s.startsWith("/storage/")) {
        return `${window.location.origin}${s}`;
    }

    // Otherwise treat it as storage path
    return `${window.location.origin}/storage/${s.replace(/^\/+/, "")}`;
}

function escapeHtml(str) {
    return String(str)
        .replaceAll("&", "&amp;")
        .replaceAll("<", "&lt;")
        .replaceAll(">", "&gt;")
        .replaceAll('"', "&quot;")
        .replaceAll("'", "&#039;");
}

function escapeAttribute(str) {
    return escapeHtml(str);
}

function isLikelyImageUrl(url) {
    if (!url) return false;
    const clean = String(url).split("?")[0].toLowerCase();
    return (
        clean.endsWith(".jpg") ||
        clean.endsWith(".jpeg") ||
        clean.endsWith(".png") ||
        clean.endsWith(".gif") ||
        clean.endsWith(".webp") ||
        clean.endsWith(".svg")
    );
}

function isLikelyVideoFileUrl(url) {
    if (!url) return false;
    const clean = String(url).split("?")[0].toLowerCase();
    return (
        clean.endsWith(".mp4") ||
        clean.endsWith(".webm") ||
        clean.endsWith(".mov") ||
        clean.endsWith(".m4v") ||
        clean.endsWith(".avi") ||
        clean.endsWith(".mkv") ||
        clean.endsWith(".flv") ||
        clean.endsWith(".wmv") ||
        clean.endsWith(".3gp") ||
        clean.endsWith(".mpeg") ||
        clean.endsWith(".mpg") ||
        clean.endsWith(".ogg")
    );
}

function isTikTokUrl(url) {
    if (!url) return false;
    try {
        const host = new URL(String(url)).hostname.toLowerCase();
        return host.endsWith("tiktok.com") || host.endsWith("tiktokcdn.com");
    } catch {
        return false;
    }
}

function isYouTubeUrl(url) {
    if (!url) return false;
    try {
        const urlStr = String(url).toLowerCase();
        return urlStr.includes("youtube.com") || urlStr.includes("youtu.be");
    } catch {
        return false;
    }
}

function extractYouTubeId(url) {
    if (!url) return null;
    try {
        const urlStr = String(url);
        // Match youtube.com/watch?v=VIDEO_ID
        let match = urlStr.match(/[?&]v=([^&]+)/);
        if (match) return match[1];
        
        // Match youtu.be/VIDEO_ID
        match = urlStr.match(/youtu\.be\/([^?&]+)/);
        if (match) return match[1];
        
        // Match youtube.com/embed/VIDEO_ID
        match = urlStr.match(/youtube\.com\/embed\/([^?&]+)/);
        if (match) return match[1];
        
        return null;
    } catch {
        return null;
    }
}
