const API_BASE = window.location.origin;
const API_URL = `${API_BASE}/api/vehicles?featured=true&limit=6`;

function resolveImageUrl(pathOrUrl) {
    if (!pathOrUrl) return "";
    if (pathOrUrl.startsWith("http://") || pathOrUrl.startsWith("https://"))
        return pathOrUrl;
    if (pathOrUrl.startsWith("/storage/")) return `${API_BASE}${pathOrUrl}`;
    const cleaned = pathOrUrl.replace(/^storage\//, "").replace(/^\/+/g, "");
    return `${API_BASE}/storage/${cleaned}`;
}

async function loadVehicles() {
    try {
        // Use cached data if available
        let vehicles;
        if (window.HOMEPAGE_DATA && window.HOMEPAGE_DATA.loaded && window.HOMEPAGE_DATA.vehicles.length > 0) {
            vehicles = window.HOMEPAGE_DATA.vehicles;
        } else {
            const response = await fetch(API_URL);
            
            if (!response.ok) {
                console.error('Failed to fetch vehicles:', response.status);
                return;
            }
            
            vehicles = await response.json();
        }

        if (!Array.isArray(vehicles) || vehicles.length === 0) {
            console.warn('No vehicles found');
            return;
        }

        const currencySymbol =
            window.Currency && window.Currency.getSymbol
                ? await window.Currency.getSymbol()
                : "Rs.";

        const slider = document.getElementById("vehicleSlider");
        if (!slider) {
            console.error('Vehicle slider element not found');
            return;
        }
        
        slider.innerHTML = "";

        vehicles.forEach((vehicle) => {
            const categoryLabel =
                (vehicle.category && vehicle.category.name) ||
                vehicle.category_name ||
                vehicle.class ||
                "";
            const imageUrl = resolveImageUrl(vehicle.image_url || vehicle.image);

            // Collect all images: main image + feature images
            const allImages = [imageUrl];
            if (vehicle.feature_images && vehicle.feature_images.length > 0) {
                vehicle.feature_images.forEach((img) => {
                    const featureImgUrl = resolveImageUrl(
                        img.image_url || img.image
                    );
                    if (featureImgUrl) allImages.push(featureImgUrl);
                });
            }

            const imagesJson = JSON.stringify(allImages).replace(/"/g, "&quot;");

            slider.innerHTML += `
        <div class="swiper-slide">
            <div class="perfect-fleet-item">

                <div class="image-box" onclick="cycleVehicleImage(this)" style="cursor: pointer;" data-images='${imagesJson}' data-current-index="0">
                    <img src="${imageUrl}" alt="${vehicle.name}">
                </div>

                <div class="perfect-fleet-content">
                    <div class="perfect-fleet-title">
                        <h3>${categoryLabel}</h3>
                        <h2>${vehicle.name}</h2>
                    </div>

                    <div class="perfect-fleet-body">
                        <ul>
              <li><img src="images/icon-fleet-list-1.svg" alt="">${
                  vehicle.passengers ?? "-"
              } Passenger</li>
              <li><img src="images/icon-fleet-list-2.svg" alt="">${
                  vehicle.km ?? "-"
              } Km</li>
              <li><img src="images/icon-fleet-list-3.svg" alt="">${
                  vehicle.cc ?? "-"
              } Cc</li>
              <li><img src="images/icon-fleet-list-4.svg" alt="">Fuel: ${
                  vehicle.fuel ?? "-"
              }</li>
            </ul>
                    </div>

                    <div class="perfect-fleet-footer">
                            <h4>${currencySymbol} ${vehicle.price}</h4>
                        <div style="float: right; width: 50px; margin-left: auto;">
                        <a href="/cars/single?id=${
                            vehicle.id
                        }" class="section-icon-btn">
                            <img src="images/arrow-white.svg">
                        </a>
                        </div>
                    </div>
                </div>

            </div>
        </div>`;
        });

        // Initialize or reinitialize the Swiper after loading vehicles
        initVehicleSwiper();

        // Reinitialize WOW animations if available
        if (typeof WOW !== 'undefined') {
            new WOW().init();
        }
    } catch (error) {
        console.error('Error loading vehicles:', error);
    }
}

// Function to initialize vehicle Swiper
function initVehicleSwiper() {
    const swiperEl = document.querySelector('.car-details-slider .swiper');
    if (!swiperEl) {
        console.error('Swiper element not found');
        return;
    }

    // Destroy existing Swiper instance if it exists
    if (swiperEl.swiper) {
        try {
            swiperEl.swiper.destroy(true, true);
        } catch (e) {
            console.error('Error destroying swiper:', e);
        }
    }

    // Initialize new Swiper
    new Swiper(swiperEl, {
        slidesPerView: 1,
        speed: 1000,
        spaceBetween: 30,
        loop: true,
        autoplay: {
            delay: 3000,
        },
        navigation: {
            nextEl: '.car-button-next',
            prevEl: '.car-button-prev',
        },
        breakpoints: {
            768: {
                slidesPerView: 2,
            },
            991: {
                slidesPerView: 3,
            },
            1300: {
                slidesPerView: 4,
            },
            1600: {
                slidesPerView: 5,
            }
        }
    });
}

// Function to cycle through vehicle images on click
function cycleVehicleImage(imageBox) {
    const images = JSON.parse(imageBox.getAttribute("data-images"));
    let currentIndex = parseInt(imageBox.getAttribute("data-current-index"));

    // Move to next image
    currentIndex = (currentIndex + 1) % images.length;

    // Update image
    const img = imageBox.querySelector("img");
    img.src = images[currentIndex];

    // Save current index
    imageBox.setAttribute("data-current-index", currentIndex);
}

document.addEventListener("DOMContentLoaded", loadVehicles);
