// public/js/testimonials-services.js

const TESTIMONIALS_API_URL = `${window.location.origin}/api/testimonials`;

document.addEventListener("DOMContentLoaded", async () => {
    const wrapper = document.getElementById("testimonialWrapper");
    if (!wrapper) return;

    try {
        const res = await fetch(TESTIMONIALS_API_URL);
        if (!res.ok) throw new Error("API error: " + res.status);

        const testimonials = await res.json();
        wrapper.innerHTML = "";

        if (!Array.isArray(testimonials) || testimonials.length === 0) {
            wrapper.innerHTML = `<div class="swiper-slide"><p>No testimonials available.</p></div>`;
            return;
        }

        testimonials.forEach((testimonial) => {
            const rating = testimonial.rating || 5;
            const name = testimonial.name || "Anonymous";
            const designation = testimonial.designation || "Customer";
            const review = testimonial.review || "";
            const image = testimonial.image || "/images/author-1.jpg";

            // Generate star rating HTML
            let starsHtml = "";
            for (let i = 1; i <= 5; i++) {
                if (i <= rating) {
                    starsHtml += '<i class="fa-solid fa-star"></i>';
                } else {
                    starsHtml += '<i class="fa-regular fa-star"></i>';
                }
            }

            // Build image URL
            let imageUrl = image;
            if (image && !image.startsWith("http://") && !image.startsWith("https://")) {
                imageUrl = `/storage/${image.replace(/^\/+/, "")}`;
            }

            const slideHtml = `
                <div class="swiper-slide">
                    <div class="testimonial-item">
                        <div class="testimonial-header">
                            <div class="testimonial-rating">
                                ${starsHtml}
                            </div>
                            <div class="testimonial-content">
                                <p>${escapeHtml(review)}</p>
                            </div>
                        </div>
                        <div class="testimonial-body">
                            <div class="author-image">
                                <figure class="image-anime">
                                    <img src="${escapeAttribute(imageUrl)}" alt="${escapeAttribute(name)}" onerror="this.src='/images/author-1.jpg'">
                                </figure>
                            </div>
                            <div class="author-content">
                                <h3>${escapeHtml(name)}</h3>
                                <p>${escapeHtml(designation)}</p>
                            </div>
                        </div>
                    </div>
                </div>
            `;

            wrapper.insertAdjacentHTML("beforeend", slideHtml);
        });

        // Re-initialize Swiper after loading testimonials
        if (typeof Swiper !== "undefined") {
            setTimeout(() => {
                const testimonialSwiper = new Swiper(".testimonial-slider .swiper", {
                    loop: true,
                    speed: 1000,
                    slidesPerView: 1,
                    spaceBetween: 30,
                    autoplay: {
                        delay: 5000,
                        disableOnInteraction: false,
                    },
                    breakpoints: {
                        768: {
                            slidesPerView: 2,
                        },
                        1024: {
                            slidesPerView: 3,
                        },
                    },
                });
            }, 100);
        }

        // Re-init WOW animations
        if (typeof WOW === "function") {
            new WOW().init();
        }
    } catch (err) {
        console.error("Testimonials load failed:", err);
        wrapper.innerHTML = `<div class="swiper-slide"><p>Failed to load testimonials.</p></div>`;
    }
});

function escapeHtml(str) {
    if (!str) return "";
    return String(str)
        .replaceAll("&", "&amp;")
        .replaceAll("<", "&lt;")
        .replaceAll(">", "&gt;")
        .replaceAll('"', "&quot;")
        .replaceAll("'", "&#039;");
}

function escapeAttribute(str) {
    return escapeHtml(str);
}
