// testimonial.js
// Fetches testimonials from Laravel API and renders them into the testimonials page.

function escapeHtml(value) {
    return String(value ?? "")
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/\"/g, "&quot;")
        .replace(/'/g, "&#039;");
}

function renderStars(rating) {
    const r = Math.max(0, Math.min(5, Number(rating) || 0));
    let html = "";
    for (let i = 1; i <= 5; i += 1) {
        html +=
            i <= r
                ? '<i class="fa-solid fa-star"></i>'
                : '<i class="fa-regular fa-star"></i>';
    }
    return html;
}

function imageUrl(path) {
    if (!path) return "images/author-1.jpg";
    // Filament FileUpload stores paths like: testimonials/xxx.jpg on the public disk
    if (/^https?:\/\//i.test(path)) return path;
    return `/storage/${String(path).replace(/^\/+/, "")}`;
}

async function fetchTestimonials() {
    const response = await fetch("/api/testimonials", {
        headers: { Accept: "application/json" },
    });

    if (!response.ok) return [];

    const items = await response.json();
    return Array.isArray(items) ? items : [];
}

function renderHomeSlides(items) {
    return items
        .map((t) => {
            return `
    <div class="swiper-slide">
      <div class="testimonial-item">
        <div class="testimonial-header">
          <div class="testimonial-rating">
            ${renderStars(t.rating)}
          </div>
          <div class="testimonial-content">
            <p>${escapeHtml(t.review)}</p>
          </div>
        </div>
        <div class="testimonial-body">
          <div class="author-image">
            <figure class="image-anime">
              <img src="${imageUrl(t.image)}" alt="">
            </figure>
          </div>
          <div class="author-content">
            <h3>${escapeHtml(t.name)}</h3>
            <p>${escapeHtml(t.designation)}</p>
          </div>
        </div>
      </div>
    </div>
    `;
        })
        .join("");
}

function initOrReinitTestimonialSwiper(root, itemCount) {
    if (!window.Swiper) return;

    const slider = root.closest(".testimonial-slider") || root;
    const swiperEl = slider.querySelector(".swiper");
    if (!swiperEl) return;

    // If function.js already initialized Swiper, destroy it so we can pick up the new slides.
    if (swiperEl.swiper) {
        try {
            swiperEl.swiper.destroy(true, true);
        } catch (_) {
            // ignore
        }
    }

    const nextEl = slider.querySelector(".testimonial-button-next");
    const prevEl = slider.querySelector(".testimonial-button-prev");
    const paginationEl = slider.querySelector(".swiper-pagination");

    const count = Number(itemCount) || 0;

    const options = {
        slidesPerView: 1,
        speed: 1000,
        spaceBetween: 30,
        loop: count > 1,
        // Keep arrows visible even if there aren't enough slides.
        watchOverflow: false,
        autoplay: {
            delay: 3000,
        },
        breakpoints: {
            768: {
                slidesPerView: 2,
            },
            991: {
                slidesPerView: 3,
            },
        },
    };

    if (paginationEl) {
        options.pagination = {
            el: paginationEl,
            clickable: true,
        };
    }

    if (nextEl || prevEl) {
        options.navigation = {
            nextEl,
            prevEl,
        };
    }

    // eslint-disable-next-line no-new
    new window.Swiper(swiperEl, options);
}

async function loadTestimonials() {
    try {
        const gridContainer = document.getElementById("testimonials-list");
        const swiperWrappers = Array.from(
            document.querySelectorAll(
                ".our-testimonial .testimonial-slider .swiper-wrapper"
            )
        );

        if (!gridContainer && swiperWrappers.length === 0) return;

        const items = await fetchTestimonials();
        if (!items.length) return;

        if (gridContainer) {
            const html = items
                .map((t, index) => {
                    const delay = (index % 9) * 0.2;
                    return `
        <div class="col-lg-4 col-md-6">
        <div class="testimonial-item page-testimonial-box wow fadeInUp" data-wow-delay="${delay}s">
          <div class="testimonial-header">
          <div class="testimonial-rating">
            ${renderStars(t.rating)}
          </div>
          <div class="testimonial-content">
            <p>${escapeHtml(t.review)}</p>
          </div>
          </div>
          <div class="testimonial-body">
          <div class="author-image">
            <figure class="image-anime">
            <img src="${imageUrl(t.image)}" alt="">
            </figure>
          </div>
          <div class="author-content">
            <h3>${escapeHtml(t.name)}</h3>
            <p>${escapeHtml(t.designation)}</p>
          </div>
          </div>
        </div>
        </div>
      `;
                })
                .join("");

            gridContainer.innerHTML = html;
        }

        if (swiperWrappers.length) {
            const slidesHtml = renderHomeSlides(items);
            swiperWrappers.forEach((wrapper) => {
                wrapper.innerHTML = slidesHtml;
                initOrReinitTestimonialSwiper(wrapper, items.length);
            });
        }

        // Re-init WOW after injecting new elements (if WOW is present)
        if (window.WOW) {
            try {
                new window.WOW().init();
            } catch (_) {
                // ignore
            }
        }
    } catch (error) {
        console.error("Error loading testimonials:", error);
    }
}

if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", loadTestimonials);
} else {
    loadTestimonials();
}
