// js/service-single.js

const BASE_URL = window.location.origin;
const API_URL = `${BASE_URL}/api/services`;

document.addEventListener("DOMContentLoaded", async () => {
    const id = new URLSearchParams(window.location.search).get("id");

    const titleEl = document.querySelector(".page-header-box h1");
    const activeCrumb = document.querySelector(".breadcrumb-item.active");

    const featuredImg = document.getElementById("serviceFeaturedImage");
    const entry = document.getElementById("serviceEntry");
    const gallery = document.getElementById("serviceGallery");
    const faqWrap = document.getElementById("rentalaccordion");
    const sidebar = document.getElementById("serviceSidebarList");

    if (!id) {
        if (titleEl) titleEl.textContent = "Service Not Found";
        if (entry) entry.innerHTML = "<p>No service ID in URL.</p>";
        return;
    }

    try {
        // Load all services (your API returns list)
        const res = await fetch(API_URL);
        if (!res.ok) throw new Error("API error: " + res.status);

        const services = await res.json();
        const service = Array.isArray(services)
            ? services.find((x) => String(x.id) === String(id))
            : null;

        if (!service) {
            if (titleEl) titleEl.textContent = "Service Not Found";
            if (entry)
                entry.innerHTML =
                    "<p>The requested service could not be found.</p>";
            return;
        }

        // Header
        if (titleEl) titleEl.textContent = service.title;
        if (activeCrumb)
            activeCrumb.textContent = (service.title || "").toLowerCase();
        document.title = `${service.title} - Finest`;

        // Featured image
        if (featuredImg) {
            const imgUrl = toStorageUrl(service.featured_image);
            if (imgUrl) featuredImg.src = imgUrl;
            featuredImg.alt = service.title || "Service";
        }

        // Main content
        if (entry) {
            entry.innerHTML = `
        <h2 class="text-anime-style-3" data-cursor="-opaque">${escapeHtml(
            service.subtitle || "Discover premium rental services"
        )}</h2>
        <p class="wow fadeInUp" data-wow-delay="0.2s">${escapeHtml(
            service.description || ""
        )}</p>
      `;
        }

        // Gallery images
        if (gallery) {
            const imgs = Array.isArray(service.gallery_images)
                ? service.gallery_images
                : [];
            gallery.innerHTML = "";

            if (imgs.length > 0) {
                imgs.forEach((path, index) => {
                    const delay = ((index + 1) * 0.2).toFixed(1);
                    const url = toStorageUrl(path);

                    gallery.insertAdjacentHTML(
                        "beforeend",
                        `
            <div class="col-lg-3 col-6">
              <div class="service-gallery wow fadeInUp" data-wow-delay="${delay}s" data-cursor-text="View">
                <a href="${url}">
                  <figure class="image-anime">
                    <img src="${url}" alt="${escapeHtml(service.title)} image ${
                            index + 1
                        }">
                  </figure>
                </a>
              </div>
            </div>
            `
                    );
                });

                // Re-init magnific popup for new gallery links
                if (window.jQuery && jQuery.fn && jQuery.fn.magnificPopup) {
                    jQuery("#serviceGallery").magnificPopup({
                        delegate: "a",
                        type: "image",
                        gallery: { enabled: true },
                    });
                }
            }
        }

        // FAQs
        if (faqWrap) {
            const faqs = Array.isArray(service.faqs) ? service.faqs : [];
            faqWrap.innerHTML = "";

            if (faqs.length > 0) {
                faqs.forEach((faq, index) => {
                    const delay = (index * 0.25).toFixed(2);
                    const open = index === 0;

                    faqWrap.insertAdjacentHTML(
                        "beforeend",
                        `
            <div class="accordion-item wow fadeInUp" data-wow-delay="${delay}s">
              <h2 class="accordion-header" id="rentalheading${index + 1}">
                <button class="accordion-button ${
                    open ? "" : "collapsed"
                }" type="button"
                  data-bs-toggle="collapse"
                  data-bs-target="#rentalcollapse${index + 1}"
                  aria-expanded="${open ? "true" : "false"}"
                  aria-controls="rentalcollapse${index + 1}">
                  ${escapeHtml(faq.question)}
                </button>
              </h2>
              <div id="rentalcollapse${
                  index + 1
              }" class="accordion-collapse collapse ${open ? "show" : ""}"
                aria-labelledby="rentalheading${
                    index + 1
                }" data-bs-parent="#rentalaccordion">
                <div class="accordion-body">
                  <p>${escapeHtml(faq.answer)}</p>
                </div>
              </div>
            </div>
            `
                    );
                });
            } else {
                faqWrap.innerHTML = `<p>No FAQs available.</p>`;
            }
        }

        // Sidebar list (all services)
        if (sidebar && Array.isArray(services)) {
            sidebar.innerHTML = "";
            services.forEach((s) => {
                const active =
                    String(s.id) === String(id) ? ' class="active"' : "";
                sidebar.insertAdjacentHTML(
                    "beforeend",
                    `<li><a href="/services/single?id=${
                        s.id
                    }"${active}>${escapeHtml(s.title)}</a></li>`
                );
            });
        }

        // Re-init WOW for injected elements
        if (typeof WOW === "function") new WOW().init();
    } catch (e) {
        console.error(e);
        if (titleEl) titleEl.textContent = "Error Loading Service";
        if (entry) entry.innerHTML = "<p>Failed to load service details.</p>";
    }
});

function toStorageUrl(path) {
    if (!path) return "";
    if (path.startsWith("http://") || path.startsWith("https://")) return path;
    if (path.startsWith("/storage/")) return BASE_URL + path;
    if (path.startsWith("storage/")) return `${BASE_URL}/${path}`;
    return `${BASE_URL}/storage/${path.replace(/^\/+/, "")}`;
}

function escapeHtml(str) {
    return String(str || "")
        .replaceAll("&", "&amp;")
        .replaceAll("<", "&lt;")
        .replaceAll(">", "&gt;")
        .replaceAll('"', "&quot;")
        .replaceAll("'", "&#039;");
}
