(function () {
    "use strict";

    let allCategories = [];
    let allVehicles = [];
    let allMakes = [];
    let currentMake = null;

    function getApiBase() {
        return window.location.origin;
    }

    async function loadPopularMakes(selectedMakeId = null) {
        try {
            const apiBase = getApiBase();

            // Load data if not already loaded
            if (
                allVehicles.length === 0 ||
                allCategories.length === 0 ||
                allMakes.length === 0
            ) {
                const [categoriesResponse, vehiclesResponse, makesResponse] =
                    await Promise.all([
                        fetch(`${apiBase}/api/categories`),
                        fetch(`${apiBase}/api/vehicles`),
                        fetch(`${apiBase}/api/makes`),
                    ]);

                if (
                    !categoriesResponse.ok ||
                    !vehiclesResponse.ok ||
                    !makesResponse.ok
                ) {
                    throw new Error("Failed to load data");
                }

                const categoriesData = await categoriesResponse.json();
                const vehiclesData = await vehiclesResponse.json();
                const makesData = await makesResponse.json();

                allCategories = Array.isArray(categoriesData)
                    ? categoriesData
                    : categoriesData.value || [];
                allVehicles = Array.isArray(vehiclesData)
                    ? vehiclesData
                    : vehiclesData.value || [];
                allMakes = Array.isArray(makesData)
                    ? makesData
                    : makesData.value || [];

                console.log("Loaded vehicles:", allVehicles.length);
                console.log("Loaded makes:", allMakes.length);
                console.log("Loaded categories:", allCategories.length);
            }

            // Set current make (null means show all vehicles)
            if (selectedMakeId !== undefined) {
                currentMake = selectedMakeId;
            }

            // Populate make tabs (always refresh to update counts)
            populateMakeTabs();

            // Render vehicles
            await renderVehicles();
        } catch (error) {
            console.error("Error loading popular makes:", error);
            console.error("Error details:", error.message);
        }
    }

    function populateMakeTabs() {
        const tabsContainer = document.querySelector(".makes-tabs-container");
        if (!tabsContainer) return;

        // Get featured makes (is_popular = true), limit to 5
        const featuredMakes = allMakes.filter((m) => m.is_popular).slice(0, 5);

        if (featuredMakes.length === 0) {
            tabsContainer.innerHTML =
                '<p style="color: white;">No featured makes found.</p>';
            return;
        }

        // Always set first make as default if none selected
        if (currentMake === null && featuredMakes.length > 0) {
            currentMake = featuredMakes[0].id;
        }

        // Force a valid make if current is invalid
        const validMake = featuredMakes.find(m => m.id === currentMake);
        if (!validMake && featuredMakes.length > 0) {
            currentMake = featuredMakes[0].id;
        }

        tabsContainer.innerHTML = "";

        // Add make-specific buttons
        featuredMakes.forEach((make, index) => {
            // Count vehicles for this make
            const vehicleCount = allVehicles.filter(
                (v) => v.make_id == make.id
            ).length;

            const button = document.createElement("button");
            button.type = "button";
            button.className = "make-tab-btn";
            button.style.cssText = `
                padding: 20px 35px;
                background: ${
                    currentMake == make.id
                        ? "linear-gradient(135deg, rgba(207, 170, 97, 0.3), rgba(207, 170, 97, 0.1))"
                        : "linear-gradient(135deg, rgba(45, 45, 45, 0.8), rgba(25, 25, 25, 0.9))"
                };
                border: 2px solid ${
                    currentMake == make.id
                        ? "#cfaa61"
                        : "rgba(255, 255, 255, 0.1)"
                };
                color: white;
                cursor: pointer;
                border-radius: 15px;
                transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
                position: relative;
                overflow: hidden;
                backdrop-filter: blur(10px);
                box-shadow: ${
                    currentMake == make.id
                        ? "0 8px 32px rgba(207, 170, 97, 0.3), 0 0 0 1px rgba(207, 170, 97, 0.2)"
                        : "0 4px 20px rgba(0, 0, 0, 0.3), inset 0 1px 0 rgba(255, 255, 255, 0.1)"
                };
                transform: ${
                    currentMake == make.id
                        ? "translateY(-2px)"
                        : "translateY(0)"
                };
                min-width: 140px;
            `;

            // Add a subtle shine effect
            const shine = document.createElement("div");
            shine.style.cssText = `
                position: absolute;
                top: 0;
                left: -100%;
                width: 100%;
                height: 100%;
                background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.1), transparent);
                transition: left 0.6s;
                pointer-events: none;
            `;
            button.appendChild(shine);

            button.innerHTML += `
                <div style="text-align: center; position: relative; z-index: 2;">
                    <div style="font-size: 18px; font-weight: 600; text-transform: uppercase; letter-spacing: 1px; margin-bottom: 4px;">${make.name}</div>
                    <div style="font-size: 11px; color: #cfaa61; font-weight: 500; opacity: 0.9;">${vehicleCount} Listings Available</div>
                </div>
            `;

            button.addEventListener("click", function (e) {
                e.preventDefault();
                e.stopPropagation();
                const makeId = make.id;

                // Update active state with beautiful animations
                document.querySelectorAll(".make-tab-btn").forEach((btn) => {
                    btn.style.background =
                        "linear-gradient(135deg, rgba(45, 45, 45, 0.8), rgba(25, 25, 25, 0.9))";
                    btn.style.borderColor = "rgba(255, 255, 255, 0.1)";
                    btn.style.transform = "translateY(0)";
                    btn.style.boxShadow =
                        "0 4px 20px rgba(0, 0, 0, 0.3), inset 0 1px 0 rgba(255, 255, 255, 0.1)";
                });

                this.style.background =
                    "linear-gradient(135deg, rgba(207, 170, 97, 0.3), rgba(207, 170, 97, 0.1))";
                this.style.borderColor = "#cfaa61";
                this.style.transform = "translateY(-2px)";
                this.style.boxShadow =
                    "0 8px 32px rgba(207, 170, 97, 0.3), 0 0 0 1px rgba(207, 170, 97, 0.2)";

                // Trigger shine effect
                const shineEffect = this.querySelector("div");
                if (shineEffect) {
                    shineEffect.style.left = "100%";
                    setTimeout(() => {
                        shineEffect.style.left = "-100%";
                    }, 600);
                }

                // Load vehicles for this make
                currentMake = makeId;
                renderVehicles();
            });

            button.addEventListener("mouseenter", function () {
                if (currentMake != make.id) {
                    this.style.borderColor = "rgba(207, 170, 97, 0.6)";
                    this.style.background =
                        "linear-gradient(135deg, rgba(207, 170, 97, 0.1), rgba(45, 45, 45, 0.9))";
                    this.style.transform = "translateY(-1px)";
                    this.style.boxShadow =
                        "0 6px 24px rgba(207, 170, 97, 0.2), inset 0 1px 0 rgba(255, 255, 255, 0.1)";

                    // Trigger shine effect on hover
                    const shineEffect = this.querySelector("div");
                    if (shineEffect) {
                        shineEffect.style.left = "100%";
                        setTimeout(() => {
                            shineEffect.style.left = "-100%";
                        }, 600);
                    }
                }
            });

            button.addEventListener("mouseleave", function () {
                if (currentMake != make.id) {
                    this.style.borderColor = "rgba(255, 255, 255, 0.1)";
                    this.style.background =
                        "linear-gradient(135deg, rgba(45, 45, 45, 0.8), rgba(25, 25, 25, 0.9))";
                    this.style.transform = "translateY(0)";
                    this.style.boxShadow =
                        "0 4px 20px rgba(0, 0, 0, 0.3), inset 0 1px 0 rgba(255, 255, 255, 0.1)";
                }
            });

            tabsContainer.appendChild(button);
        });
    }

    async function renderVehicles() {
        const swiperWrapper = document.querySelector("#popularMakesSlider");

        if (!swiperWrapper) {
            console.error("Swiper wrapper not found");
            return;
        }

        console.log("Current make:", currentMake);
        console.log("Total vehicles:", allVehicles.length);

        // Get currency symbol
        const currencySymbol =
            window.Currency && window.Currency.getSymbol
                ? await window.Currency.getSymbol()
                : "LKR";

        // Filter vehicles - show all vehicles when no make selected, otherwise filter by make
        let filteredVehicles;
        if (currentMake) {
            // Try both string and number comparison for make_id
            filteredVehicles = allVehicles.filter(
                (v) =>
                    v.make_id == currentMake ||
                    String(v.make_id) === String(currentMake)
            );
            console.log(
                "Filtered by make:",
                currentMake,
                "Found:",
                filteredVehicles.length
            );
            console.log(
                "Sample vehicle make_ids:",
                allVehicles.slice(0, 3).map((v) => v.make_id)
            );

            // If no vehicles found for selected make, show all vehicles
            if (filteredVehicles.length === 0) {
                filteredVehicles = allVehicles;
                console.log(
                    "No vehicles for selected make, showing all vehicles:",
                    filteredVehicles.length
                );
            }
        } else {
            filteredVehicles = allVehicles;
            console.log("Showing all vehicles:", filteredVehicles.length);
        }

        // Clear content
        swiperWrapper.innerHTML = "";

        // Limit to 5 vehicles
        const limitedVehicles = filteredVehicles.slice(0, 5);

        // Always show vehicles if any exist, don't show "no vehicles" message
        if (limitedVehicles.length === 0 && allVehicles.length > 0) {
            // Fallback to all vehicles if somehow no vehicles in limited set
            const fallbackVehicles = allVehicles.slice(0, 5);
            console.log("Using fallback vehicles:", fallbackVehicles.length);
            limitedVehicles.push(...fallbackVehicles);
        }

        console.log("Rendering vehicles:", limitedVehicles.length);

        limitedVehicles.forEach((vehicle) => {
            const category = allCategories.find(
                (c) => c.id === vehicle.category_id
            );
            const categoryName = category ? category.name : "Vehicle";

            const vehicleName = vehicle.name;
            const imageUrl =
                vehicle.image_url ||
                `${window.location.origin}/images/luxury-collection-img-1.jpg`;

            const vehicleHtml = `
                <div class="swiper-slide">
                    <div class="perfect-fleet-item wow fadeInUp">
                        <div class="image-box">
                            <img src="${imageUrl}" alt="${vehicleName}">
                        </div>

                        <div class="perfect-fleet-content">
                            <div class="perfect-fleet-title">
                                <h3>${categoryName}</h3>
                                <h2>${vehicleName}</h2>
                            </div>

                            <div class="perfect-fleet-body">
                                <ul>
                                    <li><img src="images/icon-fleet-list-1.svg" alt=""> ${
                                        vehicle.passengers || 0
                                    } Passenger${
                vehicle.passengers !== 1 ? "s" : ""
            }</li>
                                    <li><img src="images/icon-fleet-list-2.svg" alt=""> ${
                                        vehicle.km || 0
                                    } Km</li>
                                    <li><img src="images/icon-fleet-list-3.svg" alt=""> ${
                                        vehicle.cc || 0
                                    } Cc</li>
                                    <li><img src="images/icon-fleet-list-4.svg" alt=""> Fuel: ${
                                        vehicle.fuel || "N/A"
                                    }</li>
                                </ul>
                            </div>

                            <div class="perfect-fleet-footer">
                                <h4>${currencySymbol} ${
                vehicle.price || 0
            }</h4>
                                <div style="float: right; width: 50px; margin-left: auto;">
                                    <a href="/cars/${
                                        vehicle.id
                                    }" class="section-icon-btn">
                                        <img src="images/arrow-white.svg">
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `;

            swiperWrapper.innerHTML += vehicleHtml;
        });

        // Initialize Swiper
        initMakesSlider();

        // Reinitialize animations
        new WOW().init();
    }

    let makesSwiper = null;

    function initMakesSlider() {
        const swiperEl = document
            .querySelector("#popularMakesSlider")
            ?.closest(".swiper");
        if (!swiperEl) return;

        // Destroy existing swiper instance if it exists
        if (makesSwiper) {
            makesSwiper.destroy(true, true);
        }

        makesSwiper = new Swiper(swiperEl, {
            slidesPerView: 1,
            speed: 1000,
            spaceBetween: 30,
            loop: true,
            autoplay: {
                delay: 3500,
            },
            navigation: {
                nextEl: ".makes-button-next",
                prevEl: ".makes-button-prev",
            },
            breakpoints: {
                640: {
                    slidesPerView: 1,
                },
                768: {
                    slidesPerView: 2,
                },
                1024: {
                    slidesPerView: 4,
                },
                1280: {
                    slidesPerView: 4,
                },
            },
        });
    }

    // Load when DOM is ready
    if (document.readyState === "loading") {
        document.addEventListener("DOMContentLoaded", loadPopularMakes);
    } else {
        loadPopularMakes();
    }
})();
