// Blog API Configuration
const BLOG_API_BASE = window.location.origin;
const BLOG_API_URL = `${BLOG_API_BASE}/api/blogs`;

// Load blog articles when DOM is ready
document.addEventListener("DOMContentLoaded", () => {
    console.log("Loading blog articles from API...");
    loadBlogArticles();
});

async function loadBlogArticles() {
    try {
        console.log("Fetching blog articles from:", BLOG_API_URL);

        const response = await fetch(BLOG_API_URL);

        if (!response.ok) {
            console.error("API Error:", response.status, response.statusText);
            return;
        }

        const payload = await response.json();
        console.log("Blog API payload:", payload);

        // Supports either: [ ... ] OR { status: 'success', data: [ ... ] }
        const articles = Array.isArray(payload) ? payload : payload?.data;

        if (!Array.isArray(articles) || articles.length === 0) {
            console.warn("No articles found");
            return;
        }

        // Desktop: 4 images (1 featured + 3)
        // Phone: 2 images (1 featured + 1)
        const maxTotal = window.innerWidth < 768 ? 2 : 4;
        const maxOthers = Math.max(0, maxTotal - 1);

        // Get featured article (first one)
        const featuredArticle = articles[0];
        const otherArticles = articles.slice(1, 1 + maxOthers);

        console.log("Featured article:", featuredArticle);
        console.log("Other articles:", otherArticles.length);

        // Populate featured article
        populateFeaturedArticle(featuredArticle);

        // Populate other articles
        populateOtherArticles(otherArticles);

        // Reinitialize animations
        if (typeof WOW !== "undefined") {
            new WOW().init();
        }
    } catch (error) {
        console.error("Error loading blog articles:", error);
    }
}

function populateFeaturedArticle(article) {
    const container = document.getElementById("featured-article-container");
    if (!container) {
        console.error("Featured article container not found");
        return;
    }

    const imageUrl =
        resolveBlogImageUrl(article?.image_url || article?.image) ||
        "images/post-1.jpg";
    const title = article.title || "Untitled";
    const date = article.date ? formatDate(article.date) : "Jan 01, 2024";

    const html = `
        <div class="highlighted-article-post wow fadeInUp">
            <!-- Highlighted Article Featured Image Start -->
            <div class="highlighted-article-featured-img">
                <figure>
                    <a href="/blog/single?id=${
                        article.id
                    }" class="image-anime" data-cursor-text="View">
                        <img src="${imageUrl}" alt="${title}">
                    </a>
                </figure>
            </div>
            <!-- Highlighted Article Featured Image End -->

            <!-- Highlighted Article Body Start -->
            <div class="highlighted-article-body">
                <!-- Article Meta Start -->
                <div class="article-meta">
                    <ul>
                        <li><a href="#"><i class="fa-solid fa-calendar-days"></i> ${date}</a></li>
                    </ul>
                </div>
                <!-- Article Meta End -->

                <!-- Highlighted Article Content Start -->
                <div class="highlighted-article-content">
                    <h3><a href="/blog/single?id=${article.id}">${escapeHtml(
        title
    )}</a></h3>
                    <a href="/blog/single?id=${
                        article.id
                    }" class="section-icon-btn">
                        <img src="images/arrow-white.svg" alt="">
                    </a>
                </div>
                <!-- Highlighted Article Content End -->
            </div>
            <!-- Highlighted Article Body End -->
        </div>
    `;

    container.innerHTML = html;
    console.log("Featured article populated");
}

function populateOtherArticles(articles) {
    const container = document.getElementById("articles-container");
    if (!container) {
        console.error("Articles container not found");
        return;
    }

    let html = "";

    articles.forEach((article, index) => {
        const imageUrl =
            resolveBlogImageUrl(article?.image_url || article?.image) ||
            `images/post-${index + 2}.jpg`;
        const title = article.title || "Untitled";
        const date = article.date ? formatDate(article.date) : "Jan 01, 2024";
        const delay = (index * 0.25).toFixed(2);

        html += `
            <div class="article-post wow fadeInUp" ${
                index > 0 ? `data-wow-delay="${delay}s"` : ""
            }>
                <div class="article-featured-img">
                    <figure>
                        <a href="/blog/single?id=${
                            article.id
                        }" class="image-anime" data-cursor-text="View">
                            <img src="${imageUrl}" alt="${title}">
                        </a>
                    </figure>
                </div>

                <div class="article-post-body">
                    <!-- Article Meta Start -->
                    <div class="article-meta">
                        <ul>
                            <li><a href="#"><i class="fa-solid fa-calendar-days"></i> ${date}</a></li>
                        </ul>
                    </div>
                    <!-- Article Meta End -->

                    <div class="article-post-content">
                        <h3><a href="/blog/single?id=${
                            article.id
                        }">${escapeHtml(title)}</a></h3>
                        <a href="/blog/single?id=${
                            article.id
                        }" class="read-story-btn">read story</a>
                    </div>
                </div>
            </div>
        `;
    });

    container.innerHTML = html;
    console.log("Other articles populated:", articles.length);
}

function formatDate(dateString) {
    if (!dateString) return "Jan 01, 2024";

    const date = new Date(dateString);
    const options = { year: "numeric", month: "short", day: "2-digit" };
    return date.toLocaleDateString("en-US", options).toLowerCase();
}

function escapeHtml(text) {
    const div = document.createElement("div");
    div.textContent = text;
    return div.innerHTML;
}

function resolveBlogImageUrl(pathOrUrl) {
    if (!pathOrUrl) return "";
    if (typeof pathOrUrl !== "string") return "";
    if (pathOrUrl.startsWith("http://") || pathOrUrl.startsWith("https://"))
        return pathOrUrl;
    if (pathOrUrl.startsWith("/storage/"))
        return `${BLOG_API_BASE}${pathOrUrl}`;
    const cleaned = pathOrUrl.replace(/^storage\//, "").replace(/^\/+/, "");
    return `${BLOG_API_BASE}/storage/${cleaned}`;
}
