(function () {
    function getApiBase() {
        return window.location.origin;
    }

    function setSelectOptions(select, options, placeholderText) {
        if (!select) return;

        const placeholder = document.createElement("option");
        placeholder.value = "";
        placeholder.disabled = true;
        placeholder.selected = true;
        placeholder.textContent = placeholderText;

        select.innerHTML = "";
        select.appendChild(placeholder);

        options.forEach((opt) => {
            const option = document.createElement("option");
            option.value = String(opt.value);
            option.textContent = String(opt.label);
            select.appendChild(option);
        });
    }

    function uniqSorted(nums, desc = false) {
        const unique = Array.from(
            new Set(nums.filter((n) => n !== null && n !== undefined))
        );
        unique.sort((a, b) => {
            const an = Number(a);
            const bn = Number(b);
            if (Number.isNaN(an) || Number.isNaN(bn)) return 0;
            return desc ? bn - an : an - bn;
        });
        return unique;
    }

    function parsePriceRange(value) {
        if (!value) return null;
        const str = String(value).trim();
        if (!str) return null;

        if (str.endsWith("+")) {
            const min = parseInt(str.replace("+", ""), 10);
            return Number.isFinite(min) ? { min } : null;
        }

        const [minStr, maxStr] = str.split("-");
        const min = parseInt(minStr, 10);
        const max = parseInt(maxStr, 10);
        if (!Number.isFinite(min)) return null;
        if (!Number.isFinite(max)) return { min };
        return { min, max };
    }

    async function fetchJson(url) {
        const res = await fetch(url);
        if (!res.ok) throw new Error(`API error: ${res.status}`);
        return await res.json();
    }

    document.addEventListener("DOMContentLoaded", async () => {
        const form = document.getElementById("homeVehicleSearchForm");
        if (!form) return;

        const categorySelect = document.getElementById("homeVehicleType");
        const makeSelect = document.getElementById("homeMake");
        const yearSelect = document.getElementById("homeYear");
        const conditionSelect = document.getElementById("homeCondition");

        const apiBase = getApiBase();

        try {
            let categories, makes, vehicles;

            // Use cached data if available from fast-loader
            if (window.HOMEPAGE_DATA && window.HOMEPAGE_DATA.loaded) {
                categories = window.HOMEPAGE_DATA.categories || [];
                makes = window.HOMEPAGE_DATA.makes || [];
                vehicles = window.HOMEPAGE_DATA.allVehicles || window.HOMEPAGE_DATA.vehicles || [];
                console.log('✅ Using cached data for search filters');
            } else {
                // Fallback: fetch data if cache not available
                [categories, makes, vehicles] = await Promise.all([
                    fetchJson(`${apiBase}/api/categories`).catch(() => []),
                    fetchJson(`${apiBase}/api/makes`).catch(() => []),
                    fetchJson(`${apiBase}/api/vehicles`).catch(() => []),
                ]);
            }

            if (Array.isArray(categories)) {
                setSelectOptions(
                    categorySelect,
                    categories.map((c) => ({
                        value: c.id,
                        label: c.name || "Category",
                    })),
                    "Choose Vehicle Type"
                );
            }

            if (Array.isArray(makes)) {
                setSelectOptions(
                    makeSelect,
                    makes.map((m) => ({
                        value: m.id || m,
                        label: m.name || m,
                    })),
                    "Select Make"
                );
            }

            if (Array.isArray(vehicles)) {
                const years = uniqSorted(
                    vehicles
                        .map((v) => v && v.register_year)
                        .filter(
                            (y) =>
                                y !== null &&
                                y !== undefined &&
                                String(y).trim() !== ""
                        ),
                    true
                );

                if (years.length > 0) {
                    setSelectOptions(
                        yearSelect,
                        years.map((y) => ({ value: y, label: y })),
                        "Select Year"
                    );
                }

                const conditions = Array.from(
                    new Set(
                        vehicles
                            .map((v) => v && v.condition)
                            .filter(
                                (c) =>
                                    c !== null &&
                                    c !== undefined &&
                                    String(c).trim() !== ""
                            )
                            .map((c) => String(c).trim())
                    )
                );

                if (conditions.length > 0) {
                    conditions.sort((a, b) => a.localeCompare(b));
                    setSelectOptions(
                        conditionSelect,
                        conditions.map((c) => ({ value: c, label: c })),
                        "Select Condition"
                    );
                }
            }
        } catch (e) {
            console.error("Failed to initialize home search", e);
        }

        form.addEventListener("submit", (e) => {
            e.preventDefault();

            const params = new URLSearchParams();

            const categoryId =
                categorySelect && categorySelect.value
                    ? categorySelect.value
                    : "";
            const make = makeSelect && makeSelect.value ? makeSelect.value : "";
            const year = yearSelect && yearSelect.value ? yearSelect.value : "";
            const priceRangeSelect = document.getElementById("homePriceRange");
            const priceRange =
                priceRangeSelect && priceRangeSelect.value
                    ? priceRangeSelect.value
                    : "";
            const condition =
                conditionSelect && conditionSelect.value
                    ? conditionSelect.value
                    : "";

            if (categoryId) params.set("category_id", categoryId);
            if (make) params.set("make", make);
            if (year) params.set("year", year);
            if (condition) params.set("condition", condition);

            if (priceRange) {
                params.set("price_range", priceRange);
                const parsed = parsePriceRange(priceRange);
                if (parsed && parsed.min !== undefined)
                    params.set("price_min", String(parsed.min));
                if (parsed && parsed.max !== undefined)
                    params.set("price_max", String(parsed.max));
            }

            const target = new URL(
                form.action || "/cars",
                window.location.origin
            );
            target.search = params.toString();
            window.location.href = target.toString();
        });
    });
})();
