document.addEventListener("DOMContentLoaded", () => {
    console.log("FAQ.js loaded - loading FAQs from API");
    loadFAQsFromAPI();
});

async function loadFAQsFromAPI() {
    try {
        console.log(
            `Fetching FAQ data from: ${window.location.origin}/api/services`
        );

        const response = await fetch(`${window.location.origin}/api/services`);

        if (!response.ok) {
            console.error("API Error:", response.status, response.statusText);
            return;
        }

        const data = await response.json();
        console.log("API Response:", data);

        // Get the faqs array from response
        const faqs = data?.faqs || [];

        if (!Array.isArray(faqs) || faqs.length === 0) {
            console.warn("No FAQ data found");
            return;
        }

        console.log("Total sections:", faqs.length);
        console.log("FAQs Data:", JSON.stringify(faqs, null, 2));

        // Map section IDs for accordion containers
        const sectionMap = [
            {
                id: "general_information",
                accordionId: "accordion",
                titleId: "title_general_information",
            },
            {
                id: "booking_and_reservations",
                accordionId: "accordion1",
                titleId: "title_booking_and_reservations",
            },
            {
                id: "pricing_and_payment",
                accordionId: "accordion2",
                titleId: "title_pricing_and_payment",
            },
            {
                id: "vehicle_information",
                accordionId: "accordion3",
                titleId: "title_vehicle_information",
            },
        ];

        // Populate sidebar with ALL section titles
        populateSidebar(faqs);

        // Populate each accordion with FAQ items and update titles
        faqs.forEach((section, index) => {
            if (index >= sectionMap.length) {
                console.warn(
                    `Section index ${index} exceeds sectionMap length`
                );
                return;
            }

            const sectionTitle =
                section?.title?.trim() || `Section ${index + 1}`;
            const items = section?.items || [];
            const { id: sectionId, accordionId, titleId } = sectionMap[index];

            console.log(`\nProcessing Section ${index}:`);
            console.log(`  - Title: "${sectionTitle}"`);
            console.log(`  - Items count: ${items.length}`);
            console.log(`  - Accordion ID: ${accordionId}`);
            console.log(`  - Title Element ID: ${titleId}`);

            // Update section title in main content
            const titleElement = document.getElementById(titleId);
            if (titleElement) {
                titleElement.textContent = sectionTitle;
                console.log(`  ✓ Title updated in DOM`);
            } else {
                console.error(`  ✗ Title element not found: ${titleId}`);
            }

            // Populate accordion with FAQ items
            const accordionElement = document.getElementById(accordionId);
            if (accordionElement) {
                if (items.length > 0) {
                    populateAccordion(accordionId, items);
                } else {
                    accordionElement.innerHTML =
                        '<p style="padding: 20px; color: #999; text-align: center;">No FAQs available for this section</p>';
                    console.log(`  - Empty message added`);
                }
            } else {
                console.error(
                    `  ✗ Accordion element not found: ${accordionId}`
                );
            }
        });

        // Reinitialize WOW animations for newly added elements
        if (typeof WOW !== "undefined") {
            console.log("\nReinitializing WOW animations...");
            new WOW().init();
        }

        console.log("\n✓ FAQ loading completed successfully");
    } catch (error) {
        console.error("Error loading FAQs:", error);
    }
}

function populateSidebar(faqs) {
    const sidebarUl = document.querySelector(".faq-catagery-list ul");
    if (!sidebarUl) {
        console.error("Sidebar ul element not found");
        return;
    }

    const sectionIds = [
        "general_information",
        "booking_and_reservations",
        "pricing_and_payment",
        "vehicle_information",
    ];

    let html = "";
    faqs.forEach((section, index) => {
        const title = section?.title?.trim() || `Section ${index + 1}`;
        const sectionId = sectionIds[index] || `section_${index}`;
        html += `<li><a href="#${sectionId}">${escapeHtml(title)}</a></li>`;
        console.log(`Added to sidebar: "${title}" -> #${sectionId}`);
    });

    sidebarUl.innerHTML = html;
    console.log(`✓ Sidebar populated with ${faqs.length} section links`);
}

function populateAccordion(accordionId, items) {
    const accordion = document.getElementById(accordionId);
    if (!accordion) {
        console.error(`Accordion ${accordionId} not found`);
        return;
    }

    if (!items || items.length === 0) {
        accordion.innerHTML =
            '<p style="padding: 20px; color: #999; text-align: center;">No FAQs available for this section</p>';
        return;
    }

    let html = "";
    items.forEach((faq, index) => {
        const question = faq?.question?.trim() || `Question ${index + 1}`;
        const answer = faq?.answer?.trim() || `Answer not available`;
        const headingId = `${accordionId}_heading_${index}`;
        const collapseId = `${accordionId}_collapse_${index}`;
        const isFirst = index === 0;

        html += `
      <div class="accordion-item wow fadeInUp" data-wow-delay="${
          0.25 * index
      }s">
        <h2 class="accordion-header" id="${headingId}">
          <button class="accordion-button ${
              isFirst ? "" : "collapsed"
          }" type="button"
            data-bs-toggle="collapse"
            data-bs-target="#${collapseId}"
            aria-expanded="${isFirst}"
            aria-controls="${collapseId}">
            ${escapeHtml(question)}
          </button>
        </h2>
        <div id="${collapseId}" class="accordion-collapse collapse ${
            isFirst ? "show" : ""
        }"
          aria-labelledby="${headingId}"
          data-bs-parent="#${accordionId}">
          <div class="accordion-body">
            <p>${escapeHtml(answer)}</p>
          </div>
        </div>
      </div>
    `;
    });

    accordion.innerHTML = html;
    console.log(
        `✓ Accordion ${accordionId} populated with ${items.length} FAQ items`
    );
}

function escapeHtml(text) {
    const div = document.createElement("div");
    div.textContent = text || "";
    return div.innerHTML;
}
