(function (window) {
    const API_BASE = `${window.location.origin}/api`;
    const STORAGE_KEY = "currency_current";
    const CACHE_TTL_MS = 5 * 60 * 1000;

    let inMemory = null;

    function getCached() {
        try {
            const raw = localStorage.getItem(STORAGE_KEY);
            if (!raw) return null;
            const parsed = JSON.parse(raw);
            if (!parsed || typeof parsed.symbol !== "string") return null;
            if (typeof parsed.ts !== "number") return null;
            if (Date.now() - parsed.ts > CACHE_TTL_MS) return null;
            return {
                symbol: parsed.symbol,
                code: typeof parsed.code === "string" ? parsed.code : null,
            };
        } catch {
            return null;
        }
    }

    function setCached(value) {
        try {
            localStorage.setItem(
                STORAGE_KEY,
                JSON.stringify({
                    symbol: value.symbol,
                    code: value.code,
                    ts: Date.now(),
                })
            );
        } catch {
            // ignore
        }
    }

    async function fetchCurrency() {
        const res = await fetch(`${API_BASE}/currency`, {
            headers: { Accept: "application/json" },
        });
        if (!res.ok) throw new Error(`Currency API error: ${res.status}`);
        const json = await res.json();
        const symbol = json?.data?.symbol;
        const code = json?.data?.code;
        if (typeof symbol !== "string" || !symbol.trim())
            throw new Error("Currency symbol missing");
        return {
            symbol: symbol.trim(),
            code:
                typeof code === "string" && code.trim()
                    ? code.trim().toUpperCase()
                    : null,
        };
    }

    async function getCurrencySymbol() {
        if (
            inMemory &&
            typeof inMemory.symbol === "string" &&
            inMemory.symbol.trim()
        )
            return inMemory.symbol;

        const cached = getCached();
        if (cached) {
            inMemory = cached;
            return cached.symbol;
        }

        try {
            const value = await fetchCurrency();
            inMemory = value;
            setCached(value);
            return value.symbol;
        } catch {
            // Fallback if API is down
            inMemory = { symbol: "Rs.", code: null };
            return inMemory.symbol;
        }
    }

    async function getCurrencyCode(fallback = "MSRP") {
        if (
            inMemory &&
            typeof inMemory.code === "string" &&
            inMemory.code.trim()
        )
            return inMemory.code;

        const cached = getCached();
        if (cached && typeof cached.code === "string" && cached.code.trim()) {
            inMemory = cached;
            return cached.code;
        }

        try {
            const value = await fetchCurrency();
            inMemory = value;
            setCached(value);
            return value.code || fallback;
        } catch {
            return fallback;
        }
    }

    window.Currency = window.Currency || {};
    window.Currency.getSymbol = getCurrencySymbol;
    window.Currency.getCode = getCurrencyCode;
})(window);
