const API_BASE = window.location.origin;
const VEHICLES_API_URL = `${API_BASE}/api/vehicles`;
const CATEGORIES_API_URL = `${API_BASE}/api/categories`;
const MAKES_API_URL = `${API_BASE}/api/makes`;

function parseCsvParam(value) {
    if (!value) return [];
    return String(value)
        .split(",")
        .map((v) => v.trim())
        .filter(Boolean);
}

function readInitialFiltersFromUrl() {
    const params = new URLSearchParams(window.location.search);
    return {
        categoryIds: new Set(parseCsvParam(params.get("category_id"))),
        makes: new Set(parseCsvParam(params.get("make"))),
        conditions: new Set(parseCsvParam(params.get("condition"))),
        priceRanges: new Set(parseCsvParam(params.get("price_range"))),
        years: new Set(
            parseCsvParam(params.get("year") || params.get("register_year"))
        ),
        serverFilterParams: (() => {
            const allowed = new URLSearchParams();
            const make = params.get("make");
            const condition = params.get("condition");
            const priceMin = params.get("price_min");
            const priceMax = params.get("price_max");

            if (make) allowed.set("make", make);
            if (condition) allowed.set("condition", condition);
            if (priceMin) allowed.set("price_min", priceMin);
            if (priceMax) allowed.set("price_max", priceMax);

            const query = allowed.toString();
            return query ? `?${query}` : "";
        })(),
    };
}

function resolveImageUrl(pathOrUrl) {
    if (!pathOrUrl) return "";
    if (
        String(pathOrUrl).startsWith("http://") ||
        String(pathOrUrl).startsWith("https://")
    )
        return pathOrUrl;
    if (String(pathOrUrl).startsWith("/storage/"))
        return `${API_BASE}${pathOrUrl}`;
    const cleaned = String(pathOrUrl)
        .replace(/^storage\//, "")
        .replace(/^\/+/, "");
    return `${API_BASE}/storage/${cleaned}`;
}

let allVehicles = [];
let selectedCategoryIds = new Set();
let selectedMakes = new Set();
let selectedConditions = new Set();
let selectedPriceRanges = new Set();
let selectedYears = new Set();
let currencySymbol = "Rs.";

// Build query parameters for filters
function buildFilterQueryParams() {
    const params = new URLSearchParams();

    if (selectedMakes.size > 0) {
        params.append("make", Array.from(selectedMakes).join(","));
    }

    if (selectedConditions.size > 0) {
        params.append("condition", Array.from(selectedConditions).join(","));
    }

    // Handle price ranges
    if (selectedPriceRanges.size > 0) {
        const minPrices = [];
        const maxPrices = [];

        selectedPriceRanges.forEach((range) => {
            const [min, max] = range.split("-");
            minPrices.push(parseInt(min));
            if (max) maxPrices.push(parseInt(max));
        });

        if (minPrices.length > 0) {
            params.append("price_min", Math.min(...minPrices));
        }
        if (maxPrices.length > 0) {
            params.append("price_max", Math.max(...maxPrices));
        }
    }

    return params.toString();
}

// Fetch vehicles with filters
async function fetchVehiclesWithFilters() {
    const queryParams = buildFilterQueryParams();
    const url = queryParams
        ? `${VEHICLES_API_URL}?${queryParams}`
        : VEHICLES_API_URL;

    try {
        const response = await fetch(url);
        if (!response.ok) throw new Error(`API error: ${response.status}`);
        return await response.json();
    } catch (error) {
        console.error("Fetch error:", error);
        return [];
    }
}

function renderVehicles() {
    const container = document.getElementById("vehicles-container");
    if (!container) return;

    container.innerHTML = "";

    // Apply all filters
    let vehiclesToShow = allVehicles;

    // Filter by category
    if (selectedCategoryIds.size > 0) {
        vehiclesToShow = vehiclesToShow.filter((v) =>
            selectedCategoryIds.has(String(v.category_id))
        );
    }

    // Filter by make
    if (selectedMakes.size > 0) {
        vehiclesToShow = vehiclesToShow.filter((v) => {
            // Check both make_id and make fields
            const makeIdMatch = selectedMakes.has(String(v.make_id));
            const makeNameMatch = selectedMakes.has(v.make);
            return makeIdMatch || makeNameMatch;
        });
    }

    // Filter by condition
    if (selectedConditions.size > 0) {
        vehiclesToShow = vehiclesToShow.filter((v) =>
            selectedConditions.has(v.condition)
        );
    }

    // Filter by price range
    if (selectedPriceRanges.size > 0) {
        vehiclesToShow = vehiclesToShow.filter((v) => {
            return Array.from(selectedPriceRanges).some((range) => {
                const [min, max] = range.split("-");
                const minPrice = parseInt(min);
                const maxPrice = max ? parseInt(max) : Infinity;
                return v.price >= minPrice && v.price <= maxPrice;
            });
        });
    }

    // Filter by register year
    if (selectedYears.size > 0) {
        vehiclesToShow = vehiclesToShow.filter((v) =>
            selectedYears.has(String(v.register_year))
        );
    }

    if (!vehiclesToShow || vehiclesToShow.length === 0) {
        container.innerHTML =
            '<div class="col-lg-12"><p>No vehicles found</p></div>';
        return;
    }

    vehiclesToShow.forEach((vehicle, index) => {
        const colDiv = document.createElement("div");
        colDiv.className = "col-lg-4 col-md-6";

        const delay = (index * 0.2).toFixed(1);
        const imageUrl =
            resolveImageUrl(vehicle.image_url || vehicle.image) ||
            "images/no-car.png";
        const categoryLabel =
            (vehicle.category && vehicle.category.name) ||
            vehicle.category_name ||
            vehicle.class ||
            "Vehicle";

        colDiv.innerHTML = `
      <div class="perfect-fleet-item fleets-collection-item wow fadeInUp" data-wow-delay="${delay}s">
        <div class="image-box">
                    <img src="${imageUrl}" alt="${vehicle.name || "Vehicle"}">
        </div>
        <div class="perfect-fleet-content"> 
          <div class="perfect-fleet-title">
            <h3>${categoryLabel}</h3>
            <h2>${vehicle.name || "Unknown"}</h2>
          </div>
          <div class="perfect-fleet-body">
            <ul>
              <li><img src="images/icon-fleet-list-1.svg" alt="">${
                  vehicle.passengers ?? "-"
              } Passenger</li>
              <li><img src="images/icon-fleet-list-2.svg" alt="">${
                  vehicle.km ?? "-"
              } Km</li>
              <li><img src="images/icon-fleet-list-3.svg" alt="">${
                  vehicle.cc ?? "-"
              } Cc</li>
              <li><img src="images/icon-fleet-list-4.svg" alt="">Fuel: ${
                  vehicle.fuel ?? "-"
              }</li>
            </ul>
          </div>
          <div class="perfect-fleet-footer">
            <div class="perfect-fleet-pricing">
              <h2>${currencySymbol} ${vehicle.price || 0}<span>/day</span></h2>
            </div>
            <div class="perfect-fleet-btn">
              <a href="/cars/single?id=${
                  vehicle.id
              }" class="section-icon-btn"><img src="images/arrow-white.svg" alt=""></a>
            </div>
          </div>
        </div>
      </div>
    `;

        container.appendChild(colDiv);
    });

    const paginationDiv = document.createElement("div");
    paginationDiv.className = "col-lg-12";
    paginationDiv.innerHTML = `
    <div class="fleets-pagination wow fadeInUp" data-wow-delay="0.5s">
      <ul class="pagination">
        <li><a href="#"><i class="fa-solid fa-arrow-left-long"></i></a></li>
        <li class="active"><a href="#">1</a></li>
        <li><a href="#">2</a></li>
        <li><a href="#">3</a></li>
        <li><a href="#"><i class="fa-solid fa-arrow-right-long"></i></a></li>
      </ul>
    </div>
  `;
    container.appendChild(paginationDiv);
}

function renderCategoryFilters(categories) {
    const list = document.getElementById("category-filter-list");
    if (!list) return;

    list.innerHTML = "";
    if (!categories || categories.length === 0) {
        list.innerHTML = "<li><span>No categories found</span></li>";
        return;
    }

    categories.forEach((cat, index) => {
        const id = `category_${cat.id}`;
        const li = document.createElement("li");
        li.className = "form-check";
        li.innerHTML = `
      <input class="form-check-input" type="checkbox" value="${
          cat.id
      }" id="${id}">
      <label class="form-check-label" for="${id}">${
            cat.name || "Category"
        }</label>
    `;

        const checkbox = li.querySelector("input");
        checkbox.addEventListener("change", (e) => {
            const value = String(e.target.value);
            if (e.target.checked) {
                selectedCategoryIds.add(value);
            } else {
                selectedCategoryIds.delete(value);
            }
            renderVehicles();
        });

        list.appendChild(li);
    });
}

// Setup make filter listeners
function setupMakeFilters() {
    document.querySelectorAll(".make-filter").forEach((checkbox) => {
        checkbox.addEventListener("change", (e) => {
            if (e.target.checked) {
                selectedMakes.add(e.target.value);
            } else {
                selectedMakes.delete(e.target.value);
            }
            renderVehicles();
        });
    });
}

// Render makes filters dynamically from API
function renderMakeFilters(makes) {
    const list = document.getElementById("make-filter-list");
    if (!list) return;

    list.innerHTML = "";
    if (!makes || makes.length === 0) {
        list.innerHTML = "<li><span>No makes found</span></li>";
        return;
    }

    makes.forEach((make, index) => {
        // Handle both string array and object array
        const makeName = typeof make === "string" ? make : make.name;
        const makeId = typeof make === "string" ? make : make.id;

        const id = `make_${makeName.replace(/\s+/g, "_").toLowerCase()}`;
        const li = document.createElement("li");
        li.className = "form-check";
        li.innerHTML = `
      <input class="form-check-input make-filter" type="checkbox" value="${makeId}" id="${id}">
      <label class="form-check-label" for="${id}">${makeName}</label>
    `;

        const checkbox = li.querySelector("input");
        checkbox.addEventListener("change", (e) => {
            if (e.target.checked) {
                selectedMakes.add(e.target.value);
            } else {
                selectedMakes.delete(e.target.value);
            }
            renderVehicles();
        });

        list.appendChild(li);
    });
}

// Setup condition filter listeners
function setupConditionFilters() {
    document.querySelectorAll(".condition-filter").forEach((checkbox) => {
        checkbox.addEventListener("change", (e) => {
            if (e.target.checked) {
                selectedConditions.add(e.target.value);
            } else {
                selectedConditions.delete(e.target.value);
            }
            renderVehicles();
        });
    });
}

// Setup price range filter listeners
function setupPriceFilters() {
    document.querySelectorAll(".price-filter").forEach((checkbox) => {
        checkbox.addEventListener("change", (e) => {
            if (e.target.checked) {
                selectedPriceRanges.add(e.target.value);
            } else {
                selectedPriceRanges.delete(e.target.value);
            }
            renderVehicles();
        });
    });
}

function applyInitialFiltersToUi(initial) {
    if (!initial) return;

    selectedCategoryIds = new Set(initial.categoryIds);
    selectedMakes = new Set(initial.makes);
    selectedConditions = new Set(initial.conditions);
    selectedPriceRanges = new Set(initial.priceRanges);
    selectedYears = new Set(initial.years);

    // Check category & make checkboxes (rendered dynamically)
    document
        .querySelectorAll("#category-filter-list input[type='checkbox']")
        .forEach((el) => {
            el.checked = selectedCategoryIds.has(String(el.value));
        });

    document
        .querySelectorAll("#make-filter-list input[type='checkbox']")
        .forEach((el) => {
            el.checked = selectedMakes.has(String(el.value));
        });

    // Condition & price checkboxes (static)
    document.querySelectorAll(".condition-filter").forEach((el) => {
        el.checked = selectedConditions.has(String(el.value));
    });

    document.querySelectorAll(".price-filter").forEach((el) => {
        el.checked = selectedPriceRanges.has(String(el.value));
    });
}

const __initialFilters = readInitialFiltersFromUrl();

Promise.all([
    fetch(`${VEHICLES_API_URL}${__initialFilters.serverFilterParams}`).then(
        (r) => {
            if (!r.ok) throw new Error(`Vehicles API error: ${r.status}`);
            return r.json();
        }
    ),
    fetch(CATEGORIES_API_URL)
        .then((r) => {
            if (!r.ok) throw new Error(`Categories API error: ${r.status}`);
            return r.json();
        })
        .catch(() => []),
    fetch(MAKES_API_URL)
        .then((r) => {
            if (!r.ok) throw new Error(`Makes API error: ${r.status}`);
            return r.json();
        })
        .catch(() => []),
    window.Currency && window.Currency.getSymbol
        ? window.Currency.getSymbol()
        : Promise.resolve("Rs."),
])
    .then(([vehicles, categories, makes, symbol]) => {
        allVehicles = Array.isArray(vehicles)
            ? vehicles
            : vehicles && vehicles.value
            ? vehicles.value
            : [];
        currencySymbol =
            typeof symbol === "string" && symbol.trim() ? symbol.trim() : "Rs.";
        renderCategoryFilters(Array.isArray(categories) ? categories : []);
        renderMakeFilters(Array.isArray(makes) ? makes : []);
        setupConditionFilters();
        setupPriceFilters();
        applyInitialFiltersToUi(__initialFilters);
        renderVehicles();
    })
    .catch((error) => {
        console.error("API Error:", error);
        const container = document.getElementById("vehicles-container");
        if (container)
            container.innerHTML =
                '<div class="col-lg-12"><p>Failed to load vehicles</p></div>';
    });
