const API_BASE = window.location.origin;

document.addEventListener("DOMContentLoaded", () => {
    const urlParams = new URLSearchParams(window.location.search);
    const vehicleId = urlParams.get("id");

    if (!vehicleId) {
        showErrorMessage(
            "Vehicle not found. Please return to the vehicle listing."
        );
        return;
    }

    loadVehicleDetails(vehicleId);
    loadRelatedVehicles(vehicleId);
});

const RELATED_VEHICLES_PER_PAGE = 3;
let allRelatedVehicles = [];
let currentPage = 1;
let autoSlideInterval = null;

function resolveImageUrl(apiVehicle) {
    const pathOrUrl =
        (apiVehicle && (apiVehicle.image_url || apiVehicle.image)) || "";
    if (!pathOrUrl) return "images/no-car.png";

    const str = String(pathOrUrl);
    if (str.startsWith("http://") || str.startsWith("https://")) return str;
    if (str.startsWith("/storage/")) return `${API_BASE}${str}`;

    const cleaned = str.replace(/^storage\//, "").replace(/^\/+/, "");
    return `${API_BASE}/storage/${cleaned}`;
}

function escapeHtml(value) {
    return String(value ?? "")
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/\"/g, "&quot;")
        .replace(/'/g, "&#039;");
}

async function loadVehicleDetails(vehicleId) {
    try {
        const currencySymbol =
            window.Currency && window.Currency.getSymbol
                ? await window.Currency.getSymbol()
                : "Rs.";

        const response = await fetch(
            `${API_BASE}/api/vehicles/${encodeURIComponent(vehicleId)}`
        );
        if (!response.ok) throw new Error(`HTTP ${response.status}`);

        const vehicle = await response.json();
        
        // Debug: Log the full vehicle object
        console.log('Vehicle data:', vehicle);
        console.log('Price:', vehicle.price);
        console.log('Discount:', vehicle.discount);
        console.log('Features raw:', vehicle.features);
        console.log('Features type:', typeof vehicle.features);

        // Header
        const title = vehicle.name || "Vehicle";
        const categoryLabel =
            (vehicle.category && vehicle.category.name) ||
            vehicle.category_name ||
            vehicle.class ||
            "Vehicle";

        // Update Header Section
        const headerTitle = document.getElementById("vehicle-header-title");
        if (headerTitle) headerTitle.textContent = title;

        const headerSubtitle = document.getElementById("vehicle-header-subtitle");
        if (headerSubtitle) {
            headerSubtitle.textContent = vehicle.name || "Vehicle Details";
        }

        const breadcrumbTitle = document.getElementById("breadcrumb-title");
        if (breadcrumbTitle) breadcrumbTitle.textContent = title;

        // Update Header Price
        const headerPrice = document.getElementById("vehicle-header-price");
        if (headerPrice) {
            const price = Number(vehicle.price || 0);
            const discount = Number(vehicle.discount || 0);
            
            console.log('=== DISCOUNT DEBUG ===');
            console.log('Raw discount value:', vehicle.discount);
            console.log('Parsed discount:', discount);
            console.log('Condition check (discount > 0):', discount > 0);
            
            if (discount > 0) {
                const discountedPrice = price - (price * discount / 100);
                console.log('Showing discount! Original:', price, 'Discounted:', discountedPrice);
                headerPrice.innerHTML = `
                    <span style="text-decoration: line-through; color: #999; font-size: 0.7em; margin-right: 8px;">${currencySymbol} ${price.toLocaleString()}</span>
                    <span style="font-weight: 600;">${currencySymbol} ${Math.round(discountedPrice).toLocaleString()}</span>
                `;
            } else {
                console.log('No discount - showing regular price');
                headerPrice.textContent = `${currencySymbol} ${price.toLocaleString()}`;
            }
        }

        // Update Vehicle Badges
        const badgeYear = document.getElementById("badge-year");
        if (badgeYear) badgeYear.textContent = vehicle.register_year || "N/A";

        const badgeMileage = document.getElementById("badge-mileage");
        if (badgeMileage) badgeMileage.textContent = vehicle.km ? `${vehicle.km} km` : "0 km";

        const badgeTransmission = document.getElementById("badge-transmission");
        if (badgeTransmission) badgeTransmission.textContent = vehicle.transmission || "N/A";

        const badgeFuel = document.getElementById("badge-fuel");
        if (badgeFuel) badgeFuel.textContent = vehicle.fuel || "N/A";

        // Old title elements (keep for compatibility)
        const titleEl = document.getElementById("vehicle-title");
        if (titleEl) titleEl.textContent = title;

        const categoryEl = document.getElementById("vehicle-category");
        if (categoryEl) categoryEl.textContent = categoryLabel;

        // Sidebar details
        const priceEl = document.getElementById("vehicle-price");
        if (priceEl) {
            const currencyCode =
                window.Currency && window.Currency.getCode
                    ? await window.Currency.getCode("MSRP")
                    : "MSRP";
            const price = Number(vehicle.price || 0);
            const discount = Number(vehicle.discount || 0);
            
            if (discount > 0) {
                const discountedPrice = price - (price * discount / 100);
                priceEl.innerHTML = `
                    <span style="text-decoration: line-through; color: #999; font-size: 0.6em; margin-right: 8px;">${currencySymbol} ${price.toLocaleString()}</span>
                    ${currencySymbol} ${Math.round(discountedPrice).toLocaleString()}<span>${currencyCode}</span>
                `;
            } else {
                priceEl.innerHTML = `${currencySymbol} ${price}<span>${currencyCode}</span>`;
            }
        }

        const setText = (id, value) => {
            const el = document.getElementById(id);
            if (el) el.textContent = value ?? "-";
        };

        setText("vehicle-seating", vehicle.passengers ?? "-");
        setText("vehicle-fuel", vehicle.fuel ?? "-");
        setText("vehicle-doors", vehicle.body_type ?? "-");
        setText("vehicle-transmission", vehicle.transmission ?? "-");
        setText("vehicle-climate", vehicle.climate_control ? "Yes" : "No");
        setText(
            "vehicle-mileage",
            vehicle.mileage ? `${vehicle.mileage} km/l` : "-"
        );
        setText("vehicle-year", vehicle.register_year ?? "-");
        setText("vehicle-condition", vehicle.condition ?? "-");
        setText("vehicle-edition", vehicle.edition ?? "-");

        // Populate Car Overview Section
        setText("overview-body", vehicle.body_type ?? "-");
        setText("overview-condition", vehicle.condition ?? "-");
        setText("overview-mileage", vehicle.km ? `${vehicle.km} km` : "-");
        setText("overview-engine", vehicle.cc ? `${vehicle.cc}cc` : "-");
        setText("overview-fuel", vehicle.fuel ?? "-");
        setText("overview-doors", vehicle.passengers ? `${vehicle.passengers} Seater` : "-");
        setText("overview-year", vehicle.register_year ?? "-");
        setText("overview-cylinder", vehicle.cylinder ?? "-");
        setText("overview-transmission", vehicle.transmission ?? "-");
        setText("overview-color", vehicle.colour ?? "-");
        setText("overview-drive-type", vehicle.drive_type ?? "-");

        const hiddenId = document.getElementById("vehicle-id");
        if (hiddenId) hiddenId.value = vehicle.id;

        // Engine sound button with simple click
        const engineSoundBtn = document.getElementById("engine-sound-btn");
        if (engineSoundBtn && vehicle.sound_url) {
            engineSoundBtn.style.display = 'flex';
            
            let audioElement = null;
            
            engineSoundBtn.addEventListener('click', function() {
                if (audioElement && !audioElement.paused) {
                    // Stop if playing
                    audioElement.pause();
                    audioElement.currentTime = 0;
                    engineSoundBtn.classList.remove('playing');
                } else {
                    // Play sound
                    if (audioElement) {
                        audioElement.pause();
                        audioElement.currentTime = 0;
                    }
                    
                    audioElement = new Audio(vehicle.sound_url);
                    engineSoundBtn.classList.add('playing');
                    
                    audioElement.play().catch(err => {
                        console.error('Error playing sound:', err);
                        engineSoundBtn.classList.remove('playing');
                    });
                    
                    audioElement.onended = function() {
                        engineSoundBtn.classList.remove('playing');
                    };
                }
            });
        } else if (engineSoundBtn) {
            engineSoundBtn.style.display = 'none';
        }

        // Old engine sound button (keep for compatibility)
        const engineBtn = document.getElementById("engineSoundBtn");
        if (engineBtn && vehicle.sound_url) {
            engineBtn.dataset.soundUrl = String(vehicle.sound_url);
        }

        // Image Gallery - Main Image and Thumbnails
        const featureImages = Array.isArray(vehicle.feature_images)
            ? vehicle.feature_images
            : [];

        const mainImageUrl = resolveImageUrl(vehicle);
        const thumbnailUrl = vehicle.thumbnail_url || vehicle.thumbnail 
            ? resolveImageUrl({ image_url: vehicle.thumbnail_url, image: vehicle.thumbnail })
            : null;
        
        // Feature images - show up to 3 in grid
        const featureUrls = featureImages.slice(0, 3).map((img) =>
            resolveImageUrl({
                image_url: img.image_url,
                image: img.image,
            })
        ).filter(Boolean);

        // Set main image to thumbnail (or fallback to main vehicle image)
        const mainImage = document.getElementById("main-product-image");
        if (mainImage) {
            const displayImageUrl = thumbnailUrl || mainImageUrl;
            if (displayImageUrl) {
                mainImage.src = displayImageUrl;
                mainImage.alt = title;
            }
        }

        // Setup thumbnail grid (2x2) - only feature images, all fixed (no click)
        const thumbnailGrid = document.getElementById("thumbnail-grid");
        if (thumbnailGrid) {
            thumbnailGrid.innerHTML = featureUrls
                .map((url, idx) => `
                <div class="thumbnail-item thumbnail-fixed">
                    <img src="${url}" alt="${title} - Feature ${idx + 1}">
                </div>
            `
                )
                .join("");
        }

        // Images for swiper/slider - combine thumbnail and feature images
        const imagesToShow = [
            thumbnailUrl,
            ...featureUrls
        ].filter(Boolean);

        // Old Slider images (keep for backward compatibility)
        const swiperEl = document.querySelector(
            ".fleets-single-slider .swiper"
        );
        const wrapperEl = document.querySelector(
            ".fleets-single-slider .swiper .swiper-wrapper"
        );

        if (swiperEl && swiperEl.swiper) {
            swiperEl.swiper.destroy(true, true);
        }

        if (wrapperEl) {
            wrapperEl.innerHTML = imagesToShow
                .map(
                    (url) => `
            <div class="swiper-slide">
              <div class="fleets-slider-image">
                <figure class="image-anime">
                  <img src="${url}" alt="${title}">
                </figure>
              </div>
            </div>
          `
                )
                .join("");
        }

        if (swiperEl && typeof Swiper !== "undefined") {
            // eslint-disable-next-line no-new
            new Swiper(swiperEl, {
                loop: imagesToShow.length > 1,
                slidesPerView: 1,
                spaceBetween: 0,
                pagination: {
                    el: ".fleets-single-slider .swiper-pagination",
                    clickable: true,
                },
            });
        }

        // Amenities / features list
        const featuresEl = document.getElementById("vehicle-features");
        let features = [];
        
        // Parse features from different possible formats
        if (Array.isArray(vehicle.features)) {
            features = vehicle.features;
        } else if (typeof vehicle.features === 'string' && vehicle.features.trim()) {
            try {
                const parsed = JSON.parse(vehicle.features);
                if (Array.isArray(parsed)) {
                    features = parsed;
                }
            } catch (e) {
                console.error('Error parsing features:', e);
            }
        }
        
        console.log('Processed features:', features);
        console.log('Features length:', features.length);
        
        if (featuresEl && features.length) {
            featuresEl.innerHTML = features
                .map((f) => {
                    const featureText = typeof f === 'string' ? f : (f.feature || '');
                    return featureText ? `<li>${escapeHtml(featureText)}</li>` : '';
                })
                .filter(Boolean)
                .join("");
        }

        // Populate Features Section
        const featuresGrid = document.getElementById("features-grid");
        console.log('Features grid element:', featuresGrid);
        console.log('Will display features:', features);
        
        if (featuresGrid) {
            if (features && features.length > 0) {
                // Display features in a simple grid
                const html = features.map(f => {
                    const featureText = typeof f === 'string' ? f : (f.feature || '');
                    console.log('Processing feature:', f, 'Text:', featureText);
                    if (!featureText) return '';
                    return `
                        <div class="feature-item">
                            <span class="feature-check">✓</span>
                            ${escapeHtml(featureText)}
                        </div>
                    `;
                }).filter(Boolean).join('');

                console.log('Generated HTML:', html);
                featuresGrid.innerHTML = html || '<p style="color: rgba(255,255,255,0.6);">No features available</p>';
            } else {
                console.log('No features to display');
                featuresGrid.innerHTML = '<p style="color: rgba(255,255,255,0.6);">No features available</p>';
            }
        } else {
            console.error('Features grid element not found!');
        }

        // Populate document links
        const testDriveLink = document.getElementById('test-drive-link');
        const brochureLink = document.getElementById('brochure-link');
        const testDriveDoc = document.getElementById('test-drive-doc');
        const brochureDoc = document.getElementById('brochure-doc');
        
        const testDriveUrl = vehicle.schedule_test_drive_pdf_url || vehicle.schedule_test_drive_pdf;
        const brochureUrl = vehicle.car_brochure_pdf_url || vehicle.car_brochure_pdf;
        
        if (testDriveUrl && testDriveLink) {
            testDriveLink.href = `${API_BASE}/api/vehicles/${vehicleId}/download/test-drive`;
            testDriveLink.style.pointerEvents = 'auto';
            testDriveLink.style.opacity = '1';
        } else if (testDriveDoc) {
            testDriveDoc.style.opacity = '0.5';
            if (testDriveLink) {
                testDriveLink.style.pointerEvents = 'none';
                testDriveLink.title = 'Document not available';
            }
        }
        
        if (brochureUrl && brochureLink) {
            brochureLink.href = `${API_BASE}/api/vehicles/${vehicleId}/download/brochure`;
            brochureLink.style.pointerEvents = 'auto';
            brochureLink.style.opacity = '1';
        } else if (brochureDoc) {
            brochureDoc.style.opacity = '0.5';
            if (brochureLink) {
                brochureLink.style.pointerEvents = 'none';
                brochureLink.title = 'Document not available';
            }
        }

        document.title = `${title} - Finest`;
    } catch (error) {
        console.error("Error loading vehicle details:", error);
        showErrorMessage(
            "Failed to load vehicle details. Please try again later."
        );
    }
}

async function loadRelatedVehicles(currentVehicleId) {
    try {
        const response = await fetch(`${API_BASE}/api/vehicles`);
        if (!response.ok) throw new Error(`HTTP ${response.status}`);

        const vehicles = await response.json();
        const list = Array.isArray(vehicles)
            ? vehicles
            : vehicles && vehicles.value
            ? vehicles.value
            : [];

        allRelatedVehicles = list.filter(
            (v) => String(v.id) !== String(currentVehicleId)
        );
        currentPage = 1;

        displayRelatedVehicles();
        startAutoSlide();
    } catch (error) {
        console.error("Error loading related vehicles:", error);
    }
}

async function displayRelatedVehicles() {
    const container = document.getElementById("related-vehicles-container");
    if (!container) return;

    const currencySymbol =
        window.Currency && window.Currency.getSymbol
            ? await window.Currency.getSymbol()
            : "Rs.";

    const currencyCode =
        window.Currency && window.Currency.getCode
            ? await window.Currency.getCode("")
            : "";

    container.innerHTML = "";

    if (!allRelatedVehicles.length) {
        container.innerHTML =
            '<div class="col-lg-12"><p class="text-center">No other vehicles available.</p></div>';
        return;
    }

    const startIndex = (currentPage - 1) * RELATED_VEHICLES_PER_PAGE;
    const endIndex = startIndex + RELATED_VEHICLES_PER_PAGE;
    const vehiclesToDisplay = allRelatedVehicles.slice(startIndex, endIndex);

    vehiclesToDisplay.forEach((vehicle, index) => {
        const title = vehicle.name || "Vehicle";
        const imageUrl = resolveImageUrl(vehicle);
        const price = Number(vehicle.price || 0);
        const formattedPrice = price.toLocaleString();

        const vehicleHTML = `
      <div class="col-lg-4 col-md-6 mb-40">
        <div class="fleet-single-item">
          <div class="fleet-img">
            <img src="${imageUrl}" alt="${title}">
            <a href="/cars/single?id=${vehicle.id}" class="fleet-overlay-link"></a>
          </div>
          <div class="fleet-content">
            <div class="fleet-title">
              <h3><a href="/cars/single?id=${vehicle.id}">${title}</a></h3>
            </div>
            <div class="fleet-price">
              <span class="price-label">From</span> ${currencySymbol} ${formattedPrice} <span>LKR</span>
            </div>
            <a href="/cars/single?id=${vehicle.id}" class="btn-default">View Details</a>
          </div>
        </div>
      </div>
    `;

        container.insertAdjacentHTML("beforeend", vehicleHTML);
    });

    if (typeof WOW !== "undefined") {
        new WOW().init();
    }
}

function startAutoSlide() {
    // Clear any existing interval
    if (autoSlideInterval) {
        clearInterval(autoSlideInterval);
    }
    
    // Only start auto-slide if there are enough vehicles
    const totalPages = Math.ceil(
        allRelatedVehicles.length / RELATED_VEHICLES_PER_PAGE
    );
    
    if (totalPages <= 1) return;
    
    // Auto-slide every 5 seconds
    autoSlideInterval = setInterval(() => {
        currentPage = currentPage >= totalPages ? 1 : currentPage + 1;
        displayRelatedVehicles();
    }, 5000);
}

function stopAutoSlide() {
    if (autoSlideInterval) {
        clearInterval(autoSlideInterval);
        autoSlideInterval = null;
    }
}

function showErrorMessage(message) {
    const vehicleTitle = document.getElementById("vehicle-title");
    if (vehicleTitle) vehicleTitle.textContent = "Error";

    const contentArea = document.querySelector(".fleets-single-content");
    if (contentArea) {
        contentArea.innerHTML = `<div class="alert alert-danger">${message}</div>`;
    }
}
