// Brands.js - Fetch and Display Brands from API
const brandsConfig = {
    apiUrl: 'http://127.0.0.1:8000/api/brands',
    containerSelector: '.row',
    activeOnly: true
};

// Fetch brands from API
async function fetchBrands() {
    try {
        const url = brandsConfig.activeOnly 
            ? `${brandsConfig.apiUrl}?active=true`
            : brandsConfig.apiUrl;
        
        const response = await fetch(url);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const data = await response.json();
        
        if (data.success && data.data) {
            return data.data;
        } else {
            console.error('Invalid API response format:', data);
            return [];
        }
    } catch (error) {
        console.error('Error fetching brands:', error);
        return [];
    }
}

// Create brand logo HTML
function createBrandHTML(brand, index) {
    const delay = (index + 1) * 0.2;
    const logoUrl = brand.logo_url || '/images/icon-partners-1.svg';
    
    return `
        <div class="col-lg-3 col-6">
            <!-- Partners Logo Start -->
            <div class="partners-logo wow fadeInUp" data-wow-delay="${delay}s">
                <img src="${logoUrl}" alt="${brand.name}" onerror="this.src='/images/icon-partners-1.svg'">
            </div>
            <!-- Partners Logo End -->
        </div>
    `;
}

// Display brands in the container
function displayBrands(brands) {
    // Find the container - look for the row inside exclusive-partners section
    const section = document.querySelector('.exclusive-partners');
    if (!section) {
        console.error('Exclusive partners section not found');
        return;
    }
    
    // Find the row that contains the brand logos (not the section-row)
    const rows = section.querySelectorAll('.row');
    let container = null;
    
    // Get the second row (first row is section-row with title)
    if (rows.length > 1) {
        container = rows[1];
    }
    
    if (!container) {
        console.error('Brands container not found');
        return;
    }
    
    // Clear existing content
    container.innerHTML = '';
    
    if (brands.length === 0) {
        container.innerHTML = `
            <div class="col-12">
                <p class="text-center" style="color: #fff; padding: 40px 0;">No brands available at the moment.</p>
            </div>
        `;
        return;
    }
    
    // Add brand logos
    brands.forEach((brand, index) => {
        container.innerHTML += createBrandHTML(brand, index);
    });
    
    // Reinitialize WOW.js animations
    if (typeof WOW !== 'undefined') {
        new WOW().init();
    }
}

// Initialize brands on page load
async function initBrands() {
    console.log('Initializing brands...');
    const brands = await fetchBrands();
    console.log('Brands fetched:', brands.length);
    displayBrands(brands);
}

// Run when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initBrands);
} else {
    initBrands();
}
