document.addEventListener("DOMContentLoaded", () => {
    const API_BASE = window.location.origin;

    const imageEl = document.getElementById("post-image");
    const descriptionEl = document.getElementById("post-description");
    const tagsEl = document.getElementById("post-tags");

    if (!imageEl || !descriptionEl || !tagsEl) {
        console.error("Blog details placeholders not found");
        return;
    }

    const params = new URLSearchParams(window.location.search);
    const blogId = params.get("id");

    if (!blogId) {
        descriptionEl.innerHTML =
            "<p>Blog ID is missing. Open this page like <strong>/blog/single?id=1</strong>.</p>";
        return;
    }

    const escapeHtml = (value) => {
        const div = document.createElement("div");
        div.textContent = value == null ? "" : String(value);
        return div.innerHTML;
    };

    const normalizeTags = (tags) => {
        if (Array.isArray(tags)) {
            return tags
                .map((t) => String(t).trim())
                .filter((t) => t.length > 0);
        }

        if (typeof tags === "string") {
            return tags
                .replace(/\\/g, "")
                .split(",")
                .map((t) => t.trim())
                .filter((t) => t.length > 0);
        }

        return [];
    };

    const toImageUrl = (blog) => {
        if (blog?.image_url) return blog.image_url;
        if (blog?.image)
            return `${API_BASE}/storage/${String(blog.image).replace(
                /^\/+/,
                ""
            )}`;
        return "images/no-blog.png";
    };

    (async () => {
        try {
            const response = await fetch(
                `${API_BASE}/api/blogs/${encodeURIComponent(blogId)}`
            );
            if (!response.ok) {
                descriptionEl.innerHTML = "<p>Blog not found.</p>";
                return;
            }
            const result = await response.json();

            if (!result || result.status !== "success" || !result.data) {
                descriptionEl.innerHTML = "<p>Blog not found.</p>";
                return;
            }

            const blog = result.data;

            const socialLinks = [
                { id: "facebook-link", url: blog.facebook_url },
                { id: "youtube-link", url: blog.youtube_url },
                { id: "instagram-link", url: blog.instagram_url },
                { id: "twitter-link", url: blog.twitter_url },
            ];

            socialLinks.forEach((item) => {
                const el = document.getElementById(item.id);

                if (!el) return;

                if (item.url) {
                    el.href = item.url;
                    el.style.display = "inline-flex";
                } else {
                    // Hide icon if link not available
                    el.style.display = "none";
                }
            });

            imageEl.src = toImageUrl(blog);
            imageEl.alt = blog.title || "Blog Image";

            const descriptionHtml = (() => {
                // Preferred: safe HTML generated by backend from TipTap JSON.
                if (
                    typeof blog.description_html === "string" &&
                    blog.description_html.trim() !== ""
                ) {
                    return blog.description_html;
                }

                // Fallback: plain text.
                if (
                    typeof blog.description === "string" &&
                    blog.description.trim() !== ""
                ) {
                    return `<p>${escapeHtml(blog.description)}</p>`;
                }

                return "<p>No description available.</p>";
            })();

            descriptionEl.innerHTML = descriptionHtml;

            tagsEl.textContent = "Tags: ";

            const tags = normalizeTags(blog.tags);
            if (tags.length > 0) {
                tags.forEach((tag) => {
                    const tagEl = document.createElement("a");
                    tagEl.href = "#";
                    tagEl.textContent = tag;
                    tagEl.style.marginLeft = "8px";
                    tagsEl.appendChild(tagEl);
                });
            } else {
                tagsEl.appendChild(document.createTextNode("No tags"));
            }
        } catch (error) {
            console.error("API Error:", error);
            descriptionEl.innerHTML = "<p>Failed to load blog details.</p>";
        }
    })();
});
