/**
 * Finest Auto Dealership - Frontend API Integration
 * This file contains functions to connect the frontend with the Strapi backend API
 */

// API base URL - change this to your production URL when deploying
const API_BASE_URL = "http://localhost:1337/api";
const BASE_URL = "http://localhost:1337";

// Helper function to fetch data from API
async function fetchFromAPI(endpoint, options = {}) {
    try {
        const response = await fetch(`${API_BASE_URL}${endpoint}`, options);

        if (!response.ok) {
            throw new Error(`API error: ${response.status}`);
        }

        return await response.json();
    } catch (error) {
        console.error("API fetch error:", error);
        // showErrorMessage("Unable to load data. Please try again later.");
        return null;
    }
}

// Show error message to user
function showErrorMessage(message) {
    let errorContainer = document.getElementById("api-error-message");

    if (!errorContainer) {
        errorContainer = document.createElement("div");
        errorContainer.id = "api-error-message";
        errorContainer.className = "alert alert-danger";
        document.body.insertBefore(errorContainer, document.body.firstChild);
    }

    errorContainer.textContent = message;
    errorContainer.style.display = "block";

    setTimeout(() => {
        errorContainer.style.display = "none";
    }, 5000);
}

// Vehicle related functions
const vehicleAPI = {
    async getAllVehicles(
        page = 1,
        pageSize = 9,
        filters = {},
        sort = "createdAt:desc"
    ) {
        const queryParams = new URLSearchParams();

        queryParams.append("pagination[page]", page);
        queryParams.append("pagination[pageSize]", pageSize);
        queryParams.append("sort", sort);

        for (const [key, value] of Object.entries(filters)) {
            if (typeof value === "object") {
                for (const [operator, operatorValue] of Object.entries(value)) {
                    queryParams.append(
                        `filters[${key}][${operator}]`,
                        operatorValue
                    );
                }
            } else {
                queryParams.append(`filters[${key}]`, value);
            }
        }

        queryParams.append("populate", "mainImage");

        return await fetchFromAPI(`/vehicles?${queryParams.toString()}`);
    },

    async getVehicleBySlug(slug) {
        return await fetchFromAPI(`/vehicles/slug/${slug}`);
    },

    async getFeaturedVehicles(limit = 6) {
        return await fetchFromAPI(`/vehicles?featured=true&limit=${limit}`);
    },

    async searchVehicles(query) {
        return await fetchFromAPI(
            `/vehicles/search?query=${encodeURIComponent(query)}`
        );
    },
};

// Blog related functions
const blogAPI = {
    async getAllPosts(page = 1, pageSize = 6) {
        const queryParams = new URLSearchParams();
        queryParams.append("pagination[page]", page);
        queryParams.append("pagination[pageSize]", pageSize);
        return await fetchFromAPI(`/blogs?${queryParams.toString()}`);
    },

    async getPostBySlug(slug) {
        return await fetchFromAPI(`/blogs/slug/${slug}`);
    },

    async getFeaturedPosts() {
        return await fetchFromAPI("/blog-posts/featured");
    },

    async getPostsByCategory(categorySlug, page = 1, pageSize = 6) {
        const queryParams = new URLSearchParams();
        queryParams.append("pagination[page]", page);
        queryParams.append("pagination[pageSize]", pageSize);
        return await fetchFromAPI(
            `/blog-posts/category/${categorySlug}?${queryParams.toString()}`
        );
    },
};

// Service related functions
const serviceAPI = {
    async getAllServices() {
        return await fetchFromAPI("/services?populate=icon,featuredImage");
    },

    async getServiceBySlug(slug) {
        return await fetchFromAPI(`/services/slug/${slug}`);
    },

    async getFeaturedServices() {
        return await fetchFromAPI("/services/featured");
    },
};

// Team member related functions
const teamAPI = {
    async getAllTeamMembers() {
        return await fetchFromAPI(
            "/team-members?populate=photo&sort=displayOrder:asc"
        );
    },

    async getTeamMemberBySlug(slug) {
        return await fetchFromAPI(`/team-members/slug/${slug}`);
    },

    async getTeamMembersByDepartment(department) {
        return await fetchFromAPI(`/team-members/department/${department}`);
    },
};

// Form submission functions
const formAPI = {
    async submitInquiry(formData) {
        return await fetchFromAPI("/inquiries", {
            method: "POST",
            headers: {
                "Content-Type": "application/json",
            },
            body: JSON.stringify({
                data: formData,
            }),
        });
    },
};

// Initialize the global API object for use across the site
window.finestAPI = {
    baseUrl: BASE_URL,

    vehicle: {
        getAllVehicles: async function (filters = {}) {
            return await vehicleAPI.getAllVehicles(1, 100, filters);
        },

        getVehicleBySlug: async function (slug) {
            return await vehicleAPI.getVehicleBySlug(slug);
        },

        getFeaturedVehicles: async function (limit = 6) {
            return await vehicleAPI.getFeaturedVehicles(limit);
        },
    },

    blog: {
        getAllPosts: async function (filters = {}) {
            return await blogAPI.getAllPosts();
        },

        getPostBySlug: async function (slug) {
            return await blogAPI.getPostBySlug(slug);
        },

        getFeaturedPosts: async function () {
            return await blogAPI.getFeaturedPosts();
        },
    },

    gallery: {
        getAllImages: async function () {
            return await fetchFromAPI("/gallery-images?populate=*");
        },

        getAllVideos: async function () {
            return await fetchFromAPI("/gallery-videos?populate=*");
        },
    },

    faqs: {
        getAllFAQs: async function () {
            return await fetchFromAPI("/faqs?populate=*");
        },
    },

    form: {
        submitInquiry: async function (data) {
            return await formAPI.submitInquiry(data);
        },

        submitFinancing: async function (data) {
            return await fetchFromAPI("/financing-requests", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                body: JSON.stringify({ data }),
            });
        },

        submitContact: async function (data) {
            return await fetchFromAPI("/api/contact", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                body: JSON.stringify(data),
            });
        },
    },
};

// Example usage: Load featured vehicles on homepage
document.addEventListener("DOMContentLoaded", async () => {
    const currencySymbol =
        window.Currency && window.Currency.getSymbol
            ? await window.Currency.getSymbol()
            : "$";

    const featuredVehiclesSection = document.querySelector(
        ".featured-vehicles-section"
    );

    if (featuredVehiclesSection) {
        try {
            const featuredVehiclesContainer =
                featuredVehiclesSection.querySelector(".row");

            if (featuredVehiclesContainer) {
                featuredVehiclesContainer.innerHTML =
                    '<div class="col-12 text-center"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div></div>';

                const response =
                    await window.finestAPI.vehicle.getFeaturedVehicles();

                if (response && response.data) {
                    featuredVehiclesContainer.innerHTML = "";

                    response.data.forEach((vehicle) => {
                        const vehicleData = vehicle.attributes;
                        const imageUrl = vehicleData.mainImage?.data?.attributes
                            ?.url
                            ? `http://localhost:1337${vehicleData.mainImage.data.attributes.url}`
                            : "images/fleets-slider-img-1.jpg";

                        const vehicleCard = `
              <div class="col-lg-4 col-md-6 mb-4">
                <div class="car-item">
                  <div class="car-img">
                    <img src="${imageUrl}" alt="${vehicleData.make} ${
                            vehicleData.model
                        }" class="img-fluid">
                    ${
                        vehicleData.featured
                            ? '<span class="featured-tag">Featured</span>'
                            : ""
                    }
                  </div>
                  <div class="car-content">
                    <h4><a href="/cars/single?slug=${vehicleData.slug}">${
                            vehicleData.make
                        } ${vehicleData.model} ${vehicleData.year}</a></h4>
                    <div class="price-rating">
                      <div class="price">
                        <span>${currencySymbol}${vehicleData.price.toLocaleString()}</span>
                      </div>
                    </div>
                    <div class="car-footer">
                      <a href="/cars/single?slug=${
                          vehicleData.slug
                      }" class="btn btn-primary">View Details</a>
                    </div>
                  </div>
                </div>
              </div>
            `;

                        featuredVehiclesContainer.innerHTML += vehicleCard;
                    });
                } else {
                    featuredVehiclesContainer.innerHTML =
                        '<div class="col-12 text-center"><p>No featured vehicles available at the moment.</p></div>';
                }
            }
        } catch (error) {
            console.error("Error loading featured vehicles:", error);
        }
    }

    const contactForm = document.getElementById("contactForm");

    if (contactForm) {
        contactForm.addEventListener("submit", async function (event) {
            event.preventDefault();

            const formData = {
                name: document.getElementById("name").value,
                email: document.getElementById("email").value,
                phone: document.getElementById("phone").value,
                message: document.getElementById("message").value,
            };

            const preferredContact = document.querySelector(
                'input[name="preferredContact"]:checked'
            );
            if (preferredContact) {
                formData.preferredContact = preferredContact.value;
            }

            const purchaseTimeframe =
                document.getElementById("purchaseTimeframe");
            if (purchaseTimeframe) {
                formData.purchaseTimeframe = purchaseTimeframe.value;
            }

            const tradeIn = document.getElementById("tradeIn");
            if (tradeIn) {
                formData.tradeIn = tradeIn.checked;
            }

            const financing = document.getElementById("financing");
            if (financing) {
                formData.financing = financing.checked;
            }

            const urlParams = new URLSearchParams(window.location.search);
            const slug = urlParams.get("slug");

            if (slug) {
                formData.vehicleSlug = slug;
            }

            try {
                const submitButton = contactForm.querySelector(
                    'button[type="submit"]'
                );
                const originalButtonText = submitButton.innerHTML;
                submitButton.disabled = true;
                submitButton.innerHTML =
                    '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Sending...';

                const response = await formAPI.submitInquiry(formData);

                contactForm.reset();

                const successMessage = document.createElement("div");
                successMessage.className = "alert alert-success mt-3";
                successMessage.textContent =
                    "Thank you for your inquiry. We will contact you soon!";
                contactForm.appendChild(successMessage);

                submitButton.disabled = false;
                submitButton.innerHTML = originalButtonText;

                setTimeout(() => {
                    successMessage.remove();
                }, 5000);
            } catch (error) {
                console.error("Error submitting form:", error);

                const errorMessage = document.createElement("div");
                errorMessage.className = "alert alert-danger mt-3";
                errorMessage.textContent =
                    "There was an error submitting your inquiry. Please try again.";
                contactForm.appendChild(errorMessage);

                const submitButton = contactForm.querySelector(
                    'button[type="submit"]'
                );
                submitButton.disabled = false;
                submitButton.innerHTML = "Submit";

                setTimeout(() => {
                    errorMessage.remove();
                }, 5000);
            }
        });
    }
});

// Export API functions for use in other scripts
window.finestAPI = window.finestAPI || {};
