<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;

class Blog extends Model
{
    protected $fillable = [
        'image',
        'date',
        'title',
        'description',
        'tags',
        'facebook_url',
        'youtube_url',
        'instagram_url',
        'twitter_url',
    ];

    protected $appends = [
        'image_url',
    ];

    protected $casts = [
        'tags' => 'array',
        'description' => 'json',
        'date' => 'date',
    ];

    public function getDescriptionTextAttribute(): string
    {
        $description = $this->description;

        if (! is_array($description)) {
            return (string) ($description ?? '');
        }

        $content = $description['content'] ?? null;
        if (! is_array($content)) {
            return '';
        }

        $textParts = [];
        foreach ($content as $node) {
            $textParts[] = $this->extractTextFromNode($node);
        }

        return trim(preg_replace('/\s+/', ' ', implode(' ', array_filter($textParts))));
    }

    public function getDescriptionHtmlAttribute(): string
    {
        $description = $this->description;

        // If description is already stored as HTML/plain text, escape it and wrap.
        if (! is_array($description)) {
            $text = (string) ($description ?? '');
            $text = trim($text);

            return $text !== ''
                ? '<p>' . e($text) . '</p>'
                : '';
        }

        $content = $description['content'] ?? null;
        if (! is_array($content)) {
            return '';
        }

        $html = '';
        foreach ($content as $node) {
            $html .= $this->renderTiptapNodeToHtml($node);
        }

        return $html;
    }

    /**
     * Extract text from TipTap node
     */
    private function extractTextFromNode($node)
    {
        if (!is_array($node)) {
            return '';
        }

        $output = '';

        // Add text content
        if (isset($node['text'])) {
            $output .= $node['text'];
        }

        // Recursively process child nodes
        if (isset($node['content']) && is_array($node['content'])) {
            foreach ($node['content'] as $child) {
                $output .= $this->extractTextFromNode($child);
            }
        }

        return $output;
    }

    private function renderTiptapNodeToHtml(mixed $node): string
    {
        if (! is_array($node)) {
            return '';
        }

        $type = $node['type'] ?? null;
        $content = $node['content'] ?? null;

        $renderChildren = function () use ($content): string {
            if (! is_array($content)) {
                return '';
            }

            $out = '';
            foreach ($content as $child) {
                $out .= $this->renderTiptapNodeToHtml($child);
            }

            return $out;
        };

        if ($type === 'text') {
            $text = (string) ($node['text'] ?? '');
            $html = e($text);

            $marks = $node['marks'] ?? [];
            if (is_array($marks)) {
                foreach ($marks as $mark) {
                    if (! is_array($mark)) {
                        continue;
                    }

                    $markType = $mark['type'] ?? null;

                    if ($markType === 'bold') {
                        $html = '<strong>' . $html . '</strong>';
                    } elseif ($markType === 'italic') {
                        $html = '<em>' . $html . '</em>';
                    } elseif ($markType === 'underline') {
                        $html = '<u>' . $html . '</u>';
                    } elseif ($markType === 'strike') {
                        $html = '<s>' . $html . '</s>';
                    } elseif ($markType === 'code') {
                        $html = '<code>' . $html . '</code>';
                    } elseif ($markType === 'link') {
                        $href = $mark['attrs']['href'] ?? null;
                        $href = is_string($href) ? trim($href) : '';

                        $allowed = $href !== '' && (
                            str_starts_with($href, 'http://') ||
                            str_starts_with($href, 'https://') ||
                            str_starts_with($href, 'mailto:')
                        );

                        if ($allowed) {
                            $html = '<a href="' . e($href) . '" rel="noopener noreferrer">' . $html . '</a>';
                        }
                    }
                }
            }

            return $html;
        }

        if ($type === 'hardBreak') {
            return '<br>';
        }

        if ($type === 'horizontalRule') {
            return '<hr>';
        }

        if ($type === 'paragraph') {
            $inner = $renderChildren();
            return $inner !== '' ? '<p>' . $inner . '</p>' : '<p></p>';
        }

        if ($type === 'heading') {
            $level = (int) ($node['attrs']['level'] ?? 2);
            $level = max(1, min(6, $level));
            return '<h' . $level . '>' . $renderChildren() . '</h' . $level . '>';
        }

        if ($type === 'blockquote') {
            return '<blockquote>' . $renderChildren() . '</blockquote>';
        }

        if ($type === 'bulletList') {
            return '<ul>' . $renderChildren() . '</ul>';
        }

        if ($type === 'orderedList') {
            return '<ol>' . $renderChildren() . '</ol>';
        }

        if ($type === 'listItem') {
            return '<li>' . $renderChildren() . '</li>';
        }

        if ($type === 'codeBlock') {
            $text = '';
            if (is_array($content)) {
                foreach ($content as $child) {
                    $text .= $this->extractTextFromNode($child);
                }
            }

            return '<pre><code>' . e($text) . '</code></pre>';
        }

        // Fallback: render children (ignore unknown wrappers).
        return $renderChildren();
    }

    /**
     * Convert d/m/Y to Y-m-d before saving
     */
    public function setDateAttribute($value)
    {
        if ($value) {
            try {
                $this->attributes['date'] =
                    Carbon::createFromFormat('d/m/Y', $value)->format('Y-m-d');
            } catch (\Exception $e) {
                $this->attributes['date'] = $value;
            }
        }
    }

    /**
     * Resolve stored image path for Filament FileUpload.
     *
     * IMPORTANT: This must return a path relative to the configured disk root
     * (e.g. "blogs/filename.png" or "filename.png"), not a full URL.
     */
    public function getImageAttribute($value)
    {
        if (!$value) {
            return null;
        }

        $value = trim((string) $value);

        if (preg_match('#^https?://#i', $value)) {
            $path = parse_url($value, PHP_URL_PATH) ?: '';
            $value = basename($path);
        }

        $value = ltrim($value, '/');

        if (str_starts_with($value, 'storage/')) {
            $value = substr($value, strlen('storage/'));
        }

        $basename = basename($value);

        try {
            $disk = Storage::disk('public');

            $candidates = array_values(array_unique(array_filter([
                $value,
                'blogs/' . $basename,
                $basename,
            ])));

            foreach ($candidates as $candidate) {
                if ($disk->exists($candidate)) {
                    return $candidate;
                }
            }
        } catch (\Throwable $e) {
            // If storage isn't available for some reason, fall back to the raw path.
        }

        return $value;
    }

    /**
     * Full public image URL
     */
    public function getImageUrlAttribute(): ?string
    {
        $path = $this->image;

        if (!$path) {
            return null;
        }

        try {
            $root = app()->runningInConsole()
                ? config('app.url')
                : request()->getSchemeAndHttpHost();
        } catch (\Throwable $e) {
            $root = config('app.url');
        }

        $root = rtrim((string) $root, '/');

        return $root . '/storage/' . ltrim($path, '/');
    }
}
