<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Storage;

class VehicleFeatureImage extends Model
{
    use HasFactory;

    protected $fillable = [
        'vehicle_id',
        'image',
        'sort_order',
    ];

    protected $casts = [
        'vehicle_id' => 'integer',
        'sort_order' => 'integer',
    ];

    public function vehicle(): BelongsTo
    {
        return $this->belongsTo(Vehicle::class);
    }

    /**
     * Resolve stored image path for Filament FileUpload.
     *
     * IMPORTANT: This must return a path relative to the configured disk root
     * (e.g. "vehicle_features/filename.png" or "filename.png"), not a full URL.
     */
    public function getImageAttribute($value)
    {
        if (!$value) {
            return null;
        }

        $value = trim((string) $value);

        if (preg_match('#^https?://#i', $value)) {
            $path = parse_url($value, PHP_URL_PATH) ?: '';
            $value = basename($path);
        }

        $value = ltrim($value, '/');

        if (str_starts_with($value, 'storage/')) {
            $value = substr($value, strlen('storage/'));
        }

        $basename = basename($value);

        try {
            $disk = Storage::disk('public');

            $candidates = array_values(array_unique(array_filter([
                $value,
                'vehicle_features/' . $basename,
                'vehicles/' . $basename,
                $basename,
            ])));

            foreach ($candidates as $candidate) {
                if ($disk->exists($candidate)) {
                    return $candidate;
                }
            }
        } catch (\Throwable $e) {
            // If storage isn't available for some reason, fall back to the raw path.
        }

        return $value;
    }

    /**
     * Full public image URL
     */
    public function getImageUrlAttribute(): ?string
    {
        $path = $this->image;

        if (!$path) {
            return null;
        }

        try {
            $root = app()->runningInConsole()
                ? config('app.url')
                : request()->getSchemeAndHttpHost();
        } catch (\Throwable $e) {
            $root = config('app.url');
        }

        $root = rtrim((string) $root, '/');

        return $root . '/storage/' . ltrim($path, '/');
    }
}
