<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Storage;

class Vehicle extends Model
{
    use HasFactory;

    protected $fillable = [
        'created_by',
        'updated_by',
        'image',
        'thumbnail',
        'sound',
        'schedule_test_drive_pdf',
        'car_brochure_pdf',
        'class',
        'name',
        'colour',
        'description',
        'passengers',
        'km',
        'cc',
        'cylinder',
        'fuel',
        'price',
        'discount',
        'body_type',
        'transmission',
        'drive_type',
        'make',
        'make_id',
        'climate_control',
        'mileage',
        'register_year',
        'condition',
        'edition',
        'features',
        'tags',
        'category_id',
        'is_featured',
        'is_popular',
    ];

    protected $casts = [
        'climate_control' => 'boolean',
        'is_featured' => 'boolean',
        'is_popular' => 'boolean',
        'passengers' => 'integer',
        'km' => 'float',
        'cc' => 'integer',
        'cylinder' => 'integer',
        'price' => 'float',
        'discount' => 'float',
        'mileage' => 'float',
        'register_year' => 'integer',
        'features' => 'array',
        'tags' => 'array',
    ];

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function make()
    {
        return $this->belongsTo(\App\Models\Make::class, 'make_id');
    }

    public function featureImages(): HasMany
    {
        return $this->hasMany(VehicleFeatureImage::class)
            ->orderBy('sort_order')
            ->orderBy('id');
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Resolve stored image path for Filament FileUpload.
     *
     * IMPORTANT: This must return a path relative to the configured disk root
     * (e.g. "vehicles/filename.png" or "filename.png"), not a full URL.
     */
    public function getImageAttribute($value)
    {
        if (!$value) {
            return null;
        }

        $value = trim((string) $value);

        if (preg_match('#^https?://#i', $value)) {
            $path = parse_url($value, PHP_URL_PATH) ?: '';
            $value = basename($path);
        }

        $value = ltrim($value, '/');

        if (str_starts_with($value, 'storage/')) {
            $value = substr($value, strlen('storage/'));
        }

        $basename = basename($value);

        try {
            $disk = Storage::disk('public');

            $candidates = array_values(array_unique(array_filter([
                $value,
                'vehicles/' . $basename,
                $basename,
            ])));

            foreach ($candidates as $candidate) {
                if ($disk->exists($candidate)) {
                    return $candidate;
                }
            }
        } catch (\Throwable $e) {
            // If storage isn't available for some reason, fall back to the raw path.
        }

        return $value;
    }

    /**
     * Full public image URL
     */
    public function getImageUrlAttribute(): ?string
    {
        $path = $this->image;

        if (!$path) {
            return null;
        }

        try {
            $root = app()->runningInConsole()
                ? config('app.url')
                : request()->getSchemeAndHttpHost();
        } catch (\Throwable $e) {
            $root = config('app.url');
        }

        $root = rtrim((string) $root, '/');

        return $root . '/storage/' . ltrim($path, '/');
    }
}

