<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\VideoGallery;
use Illuminate\Http\Request;

class VideoApiController extends Controller
{
    public function index(Request $request)
    {
        $root = rtrim($request->getSchemeAndHttpHost(), '/');
        $base = $root . '/storage/';

        $limitParam = $request->query('limit');
        $limit = null;
        if ($limitParam !== null && $limitParam !== '' && $limitParam !== 'all') {
            $limit = (int) $limitParam;
            if ($limit <= 0) {
                $limit = null;
            }
        }

        $toPublicUrl = static function (?string $path) use ($root, $base): ?string {
            if (!$path) {
                return null;
            }

            if (str_starts_with($path, 'http://') || str_starts_with($path, 'https://')) {
                return $path;
            }

            if (str_starts_with($path, '/storage/')) {
                return $root . $path;
            }

            return $base . ltrim($path, '/');
        };

        $query = VideoGallery::query()
            ->select(['id', 'title', 'video', 'thumbnail', 'tiktok_url', 'is_active', 'created_at'])
            ->where('is_active', true)
            ->latest();

        if ($limit !== null) {
            $query->limit($limit);
        }

        $videos = $query
            ->get()
            ->map(function ($v) use ($toPublicUrl) {
                $videoPath = $v->video;
                
                // Use tiktok_url if video is empty
                if (empty($videoPath) && !empty($v->tiktok_url)) {
                    $videoPath = $v->tiktok_url;
                }
                
                $videoUrl = $toPublicUrl($videoPath);
                $thumbnailUrl = $v->thumbnail ? $toPublicUrl($v->thumbnail) : null;
                
                return [
                    'id' => $v->id,
                    'title' => $v->title ?? 'Untitled Video',
                    'video' => $videoUrl,
                    'thumbnail' => $thumbnailUrl,
                    'tiktok_url' => $v->tiktok_url,
                    'created_at' => optional($v->created_at)->toISOString(),
                ];
            });

        return response()->json($videos);
    }
}
