<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Vehicle;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\URL;

class VehicleApiController extends Controller
{
    private function publicImageUrl(?string $path): ?string
    {
        if (!$path) {
            return null;
        }

        if (str_starts_with($path, 'http://') || str_starts_with($path, 'https://')) {
            return $path;
        }

        $path = ltrim($path, '/');
        $path = preg_replace('#^storage/#', '', $path);

        if (!Storage::disk('public')->exists($path)) {
            return null;
        }

        return URL::to('/storage/' . $path);
    }

    private function transformVehicle(Vehicle $vehicle): Vehicle
    {
        $vehicle->image_url = $this->publicImageUrl($vehicle->image);
        $vehicle->thumbnail_url = $this->publicImageUrl($vehicle->thumbnail ?? null);
        $vehicle->sound_url = $this->publicImageUrl($vehicle->sound ?? null);
        $vehicle->schedule_test_drive_pdf_url = $this->publicImageUrl($vehicle->schedule_test_drive_pdf ?? null);
        $vehicle->car_brochure_pdf_url = $this->publicImageUrl($vehicle->car_brochure_pdf ?? null);

        // Normalize features to a flat array of strings (supports either ["a","b"] or [{feature:"a"}]).
        $rawFeatures = $vehicle->features ?? [];
        if (is_array($rawFeatures)) {
            $vehicle->features = array_values(
                array_filter(
                    array_map(function ($item) {
                        if (is_string($item)) {
                            return trim($item);
                        }
                        if (is_array($item) && isset($item['feature'])) {
                            return trim((string) $item['feature']);
                        }
                        return null;
                    }, $rawFeatures),
                    fn ($v) => is_string($v) && $v !== ''
                )
            );
        }

        if ($vehicle->relationLoaded('featureImages')) {
            $featureImages = $vehicle->featureImages
                ->take(6)
                ->values()
                ->map(function ($img) {
                    return [
                        'id' => $img->id,
                        'image' => $img->image,
                        'image_url' => $this->publicImageUrl($img->image),
                        'sort_order' => $img->sort_order,
                    ];
                })
                ->all();

            // Prevent Eloquent from serializing the relationship as "feature_images".
            $vehicle->unsetRelation('featureImages');
            $vehicle->feature_images = $featureImages;
        }

        if ($vehicle->relationLoaded('category') && $vehicle->category) {
            $vehicle->category->image_url = $this->publicImageUrl($vehicle->category->image);
            $vehicle->category_name = $vehicle->category->name;
        } else {
            $vehicle->category_name = null;
        }

        if ($vehicle->relationLoaded('make') && $vehicle->make) {
            $vehicle->make_name = $vehicle->make->name;
        } else {
            $vehicle->make_name = null;
        }

        return $vehicle;
    }

    public function index()
    {
        $query = Vehicle::select(
            'id',
            'image',
            'thumbnail',
            'sound',
            'schedule_test_drive_pdf',
            'car_brochure_pdf',
            'class',
            'name',
            'colour',
            'passengers',
            'km',
            'cc',
            'cylinder',
            'fuel',
            'price',
            'discount',
            'transmission',
            'drive_type',
            'body_type',
            'climate_control',
            'mileage',
            'register_year',
            'condition',
            'edition',
            'features',
            'tags',
            'make',
            'make_id',
            'category_id',
            'is_featured'
        )
            ->with([
                'category:id,name,image',
                'featureImages:id,vehicle_id,image,sort_order',
                'make:id,name',
            ]);

        // Filter by featured
        if (request()->has('featured') && request()->query('featured') === 'true') {
            $query->where('is_featured', true);
        }

        // Filter by make (check both make_id and make fields)
        if (request()->has('make')) {
            $makes = explode(',', request()->query('make'));
            $query->where(function($q) use ($makes) {
                $q->whereIn('make_id', $makes)
                  ->orWhereIn('make', $makes);
            });
        }

        // Filter by condition
        if (request()->has('condition')) {
            $conditions = explode(',', request()->query('condition'));
            $query->whereIn('condition', $conditions);
        }

        // Filter by price range
        if (request()->has('price_min')) {
            $query->where('price', '>=', (float) request()->query('price_min'));
        }

        if (request()->has('price_max')) {
            $query->where('price', '<=', (float) request()->query('price_max'));
        }

        // Apply limit if specified
        $limit = request()->query('limit');
        if ($limit && is_numeric($limit)) {
            $vehicles = $query->latest()->limit((int) $limit)->get();
        } else {
            $vehicles = $query->latest()->get();
        }

        $vehicles->transform(fn (Vehicle $vehicle) => $this->transformVehicle($vehicle));

        return response()->json($vehicles);
    }

    public function show(int $id)
    {
        $vehicle = Vehicle::select(
            'id',
            'image',
            'thumbnail',
            'sound',
            'schedule_test_drive_pdf',
            'car_brochure_pdf',
            'class',
            'name',
            'colour',
            'passengers',
            'km',
            'cc',
            'cylinder',
            'fuel',
            'price',
            'discount',
            'transmission',
            'drive_type',
            'body_type',
            'climate_control',
            'mileage',
            'register_year',
            'condition',
            'edition',
            'features',
            'tags',
            'make',
            'make_id',
            'category_id',
            'is_featured'
        )
            ->with([
                'category:id,name,image',
                'featureImages:id,vehicle_id,image,sort_order',
                'make:id,name',
            ])
            ->findOrFail($id);

        return response()->json($this->transformVehicle($vehicle));
    }

    public function downloadDocument(int $id, string $type)
    {
        $vehicle = Vehicle::findOrFail($id);
        
        $field = $type === 'test-drive' ? 'schedule_test_drive_pdf' : 'car_brochure_pdf';
        $filename = $vehicle->$field;
        
        if (!$filename) {
            abort(404, 'Document not found');
        }
        
        $path = storage_path('app/public/' . $filename);
        
        if (!file_exists($path)) {
            abort(404, 'File not found');
        }
        
        $documentName = $type === 'test-drive' 
            ? 'Schedule-Test-Drive.pdf' 
            : 'Car-Brochure.pdf';
        
        return response()->download($path, $documentName, [
            'Content-Type' => 'application/pdf',
        ]);
    }
}
