<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\URL;

class BrandApiController extends Controller
{
    private function publicImageUrl(?string $path): ?string
    {
        if (!$path) {
            return null;
        }

        if (str_starts_with($path, 'http://') || str_starts_with($path, 'https://')) {
            return $path;
        }

        $path = ltrim($path, '/');
        $path = preg_replace('#^storage/#', '', $path);

        if (!Storage::disk('public')->exists($path)) {
            return null;
        }

        return URL::to('/storage/' . $path);
    }

    private function transformBrand(Brand $brand): Brand
    {
        $brand->logo_url = $this->publicImageUrl($brand->logo);
        return $brand;
    }

    public function index()
    {
        $query = Brand::select(
            'id',
            'name',
            'logo',
            'is_active',
            'created_at',
            'updated_at'
        );

        // Filter by active status
        if (request()->has('active') && request()->query('active') === 'true') {
            $query->where('is_active', true);
        }

        // Apply sorting
        $query->orderBy('name', 'asc');

        // Apply limit if specified
        $limit = request()->query('limit');
        if ($limit && is_numeric($limit)) {
            $brands = $query->limit((int) $limit)->get();
        } else {
            $brands = $query->get();
        }

        $brands->transform(fn (Brand $brand) => $this->transformBrand($brand));

        return response()->json([
            'success' => true,
            'data' => $brands
        ]);
    }

    public function show(int $id)
    {
        $brand = Brand::select(
            'id',
            'name',
            'logo',
            'is_active',
            'created_at',
            'updated_at'
        )->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $this->transformBrand($brand)
        ]);
    }
}
