<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Blog;
use App\Rules\MaxWords;
use Illuminate\Http\Request;

class BlogApiController extends Controller
{
    private function normalizeTags(mixed $tags): array
    {
        if (is_array($tags)) {
            return array_values(array_filter(array_map('trim', $tags), static fn ($v) => $v !== ''));
        }

        if (! is_string($tags)) {
            return [];
        }

        // Handles values like: "tag1,tag2", "tag1,", "tag1,\\" etc.
        $clean = str_replace(["\\\"", "\\"], ['', ''], $tags);
        $parts = array_map('trim', explode(',', $clean));

        return array_values(array_filter($parts, static fn ($v) => $v !== ''));
    }

    public function index()
    {
        $posts = Blog::all();

        $posts = $posts->map(function ($post) {
            return [
                'id' => $post->id,
                'image' => $post->image,
                'image_url' => $post->image_url,
                'date' => $post->date,
                'title' => $post->title,
                'description_html' => $post->description_html,
                'description' => $post->description_text,
                // Raw JSON payload (for editors/debugging)
                'description_json' => $post->description,
                'tags' => $this->normalizeTags($post->tags),

                // Social links
                'facebook_url' => $post->facebook_url,
                'youtube_url' => $post->youtube_url,
                'instagram_url' => $post->instagram_url,
                'twitter_url' => $post->twitter_url,

                'created_at' => $post->created_at,
                'updated_at' => $post->updated_at,
            ];
        });

        return response()->json([
            'status' => 'success',
            'data' => $posts,
        ], 200);
    }

    public function show($id)
    {
        $post = Blog::find($id);

        if (! $post) {
            return response()->json([
                'status' => 'error',
                'message' => 'Blog post not found',
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'data' => [
                'id' => $post->id,
                'image' => $post->image,
                'image_url' => $post->image_url,
                'date' => $post->date,
                'title' => $post->title,
                'description_html' => $post->description_html,
                'description' => $post->description_text,
                // Raw JSON payload (for editors/debugging)
                'description_json' => $post->description,
                'tags' => $this->normalizeTags($post->tags),

                // Social links
                'facebook_url' => $post->facebook_url,
                'youtube_url' => $post->youtube_url,
                'instagram_url' => $post->instagram_url,
                'twitter_url' => $post->twitter_url,

                'created_at' => $post->created_at,
                'updated_at' => $post->updated_at,
            ],
        ], 200);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'date' => 'required|date',
            'title' => 'required|string|max:255',
            'description' => ['nullable', 'string', new MaxWords(800)],
            'tags' => 'nullable|string|max:255',

            // Social links
            'facebook_url' => 'nullable|url|max:2048',
            'youtube_url' => 'nullable|url|max:2048',
            'instagram_url' => 'nullable|url|max:2048',
            'twitter_url' => 'nullable|url|max:2048',
        ]);

        if ($request->hasFile('image')) {
            $validated['image'] = $request->file('image')->store('blogs', 'public');
        }

        $post = Blog::create($validated);

        return response()->json([
            'status' => 'success',
            'message' => 'Blog post created successfully',
            'data' => $post->fresh(),
        ], 201);
    }

    public function update(Request $request, $id)
    {
        $post = Blog::find($id);

        if (! $post) {
            return response()->json([
                'status' => 'error',
                'message' => 'Blog post not found',
            ], 404);
        }

        $validated = $request->validate([
            'image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'date' => 'sometimes|required|date',
            'title' => 'sometimes|required|string|max:255',
            'description' => ['sometimes', 'nullable', 'string', new MaxWords(800)],
            'tags' => 'nullable|string|max:255',

            // Social links
            'facebook_url' => 'nullable|url|max:2048',
            'youtube_url' => 'nullable|url|max:2048',
            'instagram_url' => 'nullable|url|max:2048',
            'twitter_url' => 'nullable|url|max:2048',
        ]);

        if ($request->hasFile('image')) {
            $validated['image'] = $request->file('image')->store('blogs', 'public');
        }

        $post->update($validated);

        return response()->json([
            'status' => 'success',
            'message' => 'Blog post updated successfully',
            'data' => $post->fresh(),
        ], 200);
    }

    public function destroy($id)
    {
        $post = Blog::find($id);

        if (! $post) {
            return response()->json([
                'status' => 'error',
                'message' => 'Blog post not found',
            ], 404);
        }

        $post->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Blog post deleted successfully',
        ], 200);
    }
}
