<?php

namespace App\Filament\Widgets;

use App\Models\Vehicle;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Filament\Widgets\TableWidget;
use Illuminate\Support\Facades\Auth;

class RecentVehicleChangesWidget extends TableWidget
{
    protected int | string | array $columnSpan = 'full';

    public static function canView(): bool
    {
        $user = Auth::user();

        if (! $user) {
            return false;
        }

        if (in_array($user->role, ['super_admin', 'admin'], true)) {
            return true;
        }

        return false;
    }

    public function table(Table $table): Table
    {
        return $table
            ->heading('Recent Vehicle Changes')
            ->query(
                Vehicle::query()
                    ->with(['creator', 'updater'])
                    ->latest('updated_at')
            )
            ->defaultPaginationPageOption(5)
            ->paginationPageOptions([5])
            ->columns([
                TextColumn::make('name')
                    ->label('Vehicle')
                    ->searchable()
                    ->wrap(),

                TextColumn::make('action')
                    ->label('Action')
                    ->badge()
                    ->state(function (Vehicle $record): string {
                        if ($record->created_at && $record->updated_at && $record->created_at->equalTo($record->updated_at)) {
                            return 'Created';
                        }

                        return 'Updated';
                    })
                    ->color(fn (string $state): string => $state === 'Created' ? 'success' : 'warning'),

                TextColumn::make('creator.name')
                    ->label('Created By')
                    ->default('—')
                    ->wrap(),

                TextColumn::make('updater.name')
                    ->label('Updated By')
                    ->default('—')
                    ->wrap(),

                TextColumn::make('updated_at')
                    ->label('Last Update')
                    ->since()
                    ->sortable(),
            ]);
    }
}
