<?php

namespace App\Filament\Widgets;

use App\Filament\Resources\Imagegalleries\ImagegalleryResource;
use App\Filament\Resources\Users\UserResource;
use App\Filament\Resources\Vehicles\VehicleResource;
use App\Filament\Resources\Videogalleries\VideogalleryResource;
use App\Models\Imagegallery;
use App\Models\User;
use App\Models\Vehicle;
use App\Models\VideoGallery;
use Filament\Widgets\Widget;
use Illuminate\Support\Facades\Auth;

class RecentActivityWidget extends Widget
{
    protected string $view = 'filament.widgets.recent-activity-widget';

    protected int | string | array $columnSpan = 1;

    public static function canView(): bool
    {
        $user = Auth::user();

        if (! $user) {
            return false;
        }

        if ($user->role === 'super_admin') {
            return true;
        }

        $options = $user->dashboard_options ?? [];

        return ! empty($options);
    }

    protected function getViewData(): array
    {
        return [
            'items' => $this->getItems(),
        ];
    }

    /**
     * @return array<int, array{type: string, title: string, url: string, at: string}>
     */
    private function getItems(): array
    {
        $user = Auth::user();
        $options = $user?->dashboard_options ?? [];

        $items = collect();

        if ($this->hasOption('vehicles')) {
            $items = $items->merge(
                Vehicle::query()
                    ->latest('updated_at')
                    ->limit(10)
                    ->get()
                    ->map(fn (Vehicle $record) => [
                        'type' => 'Vehicle',
                        'title' => (string) ($record->name ?? ('#' . $record->getKey())),
                        'url' => VehicleResource::getUrl('edit', ['record' => $record]),
                        'at' => optional($record->updated_at ?? $record->created_at)->diffForHumans() ?? '',
                        'sort' => optional($record->updated_at ?? $record->created_at)?->timestamp ?? 0,
                    ]),
            );
        }

        if ($this->hasOption('users')) {
            $items = $items->merge(
                User::query()
                    ->latest('updated_at')
                    ->limit(10)
                    ->get()
                    ->map(fn (User $record) => [
                        'type' => 'User',
                        'title' => (string) ($record->name ?? $record->email ?? ('#' . $record->getKey())),
                        'url' => UserResource::getUrl('edit', ['record' => $record]),
                        'at' => optional($record->updated_at ?? $record->created_at)->diffForHumans() ?? '',
                        'sort' => optional($record->updated_at ?? $record->created_at)?->timestamp ?? 0,
                    ]),
            );
        }

        if ($this->hasOption('image_gallery')) {
            $items = $items->merge(
                Imagegallery::query()
                    ->latest('updated_at')
                    ->limit(10)
                    ->get()
                    ->map(fn (Imagegallery $record) => [
                        'type' => 'Image',
                        'title' => (string) ($record->title ?? ('#' . $record->getKey())),
                        'url' => ImagegalleryResource::getUrl('edit', ['record' => $record]),
                        'at' => optional($record->updated_at ?? $record->created_at)->diffForHumans() ?? '',
                        'sort' => optional($record->updated_at ?? $record->created_at)?->timestamp ?? 0,
                    ]),
            );
        }

        if ($this->hasOption('video_gallery')) {
            $items = $items->merge(
                VideoGallery::query()
                    ->latest('updated_at')
                    ->limit(10)
                    ->get()
                    ->map(fn (VideoGallery $record) => [
                        'type' => 'Video',
                        'title' => (string) ($record->title ?? ('#' . $record->getKey())),
                        'url' => VideogalleryResource::getUrl('edit', ['record' => $record]),
                        'at' => optional($record->updated_at ?? $record->created_at)->diffForHumans() ?? '',
                        'sort' => optional($record->updated_at ?? $record->created_at)?->timestamp ?? 0,
                    ]),
            );
        }

        return $items
            ->sortByDesc('sort')
            ->take(10)
            ->values()
            ->map(fn (array $item) => [
                'type' => $item['type'],
                'title' => $item['title'],
                'url' => $item['url'],
                'at' => $item['at'],
            ])
            ->all();
    }

    private function hasOption(string $option): bool
    {
        $user = Auth::user();

        if (! $user) {
            return false;
        }

        if ($user->role === 'super_admin') {
            return true;
        }

        $options = $user->dashboard_options ?? [];

        return in_array($option, $options, true);
    }
}
