<?php

namespace App\Filament\Resources\Videogalleries;

use App\Filament\Resources\Videogalleries\Pages\CreateVideogallery;
use App\Filament\Resources\Videogalleries\Pages\EditVideogallery;
use App\Filament\Resources\Videogalleries\Pages\ListVideogalleries;
use App\Filament\Resources\Videogalleries\Pages\ViewVideogallery;
use App\Models\VideoGallery;
use BackedEnum;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Auth;

use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Toggle;

use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Actions\EditAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\ViewAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;

class VideogalleryResource extends Resource
{
    protected static ?string $model = VideoGallery::class;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedVideoCamera;

    protected static ?string $recordTitleAttribute = 'title';

    protected static ?string $navigationLabel = 'Video Gallery';

    protected static ?string $pluralModelName = 'Video Gallery';

    protected static ?int $navigationSort = 2;

    protected static function hasDashboardOption(): bool
    {
        $user = Auth::user();
        if ($user?->role === 'super_admin') {
            return true;
        }
        $options = $user?->dashboard_options ?? [];
        return in_array('video_gallery', $options, true);
    }

    public static function canViewAny(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canCreate(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canEdit($record): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canDelete($record): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function shouldRegisterNavigation(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function form(Schema $schema): Schema
    {
        return $schema->components([
            TextInput::make('title')
                ->required()
                ->maxLength(255),

            FileUpload::make('video')
                ->label('Video File')
                ->disk('public')
                ->directory('videogalleries/videos')
                ->maxSize(999999999)
                ->visibility('public')
                ->acceptedFileTypes([
                    'video/*',
                    'video/mp4',
                    'video/mpeg',
                    'video/quicktime',
                    'video/x-msvideo',
                    'video/x-ms-wmv',
                    'video/webm',
                    'video/x-flv',
                    'video/3gpp',
                    'video/3gpp2',
                    'video/x-matroska',
                    'application/x-mpegURL',
                    'video/MP2T',
                    'video/ogg',
                ])
                ->downloadable()
                ->openable()
                ->required(),

            FileUpload::make('thumbnail')
                ->label('Thumbnail Image')
                ->disk('public')
                ->directory('videogalleries/thumbnails')
                ->image()
                ->imageEditor()
                ->imageEditorAspectRatios([
                    '16:9',
                    '4:3',
                    '1:1',
                ])
                ->maxSize(5120)
                ->visibility('public')
                ->helperText('Upload a thumbnail image for the video (16:9 aspect ratio recommended)'),

            TextInput::make('tiktok_url')
                ->label('TikTok/YouTube URL')
                ->url()
                ->maxLength(255)
                ->helperText('Optional: Add a TikTok or YouTube URL if not uploading a video file'),

            Toggle::make('is_active')
                ->label('Show on Homepage')
                ->helperText('Toggle to show/hide this video on the homepage (maximum 4 active videos will be displayed)')
                ->default(true)
                ->inline(false),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('title')
                    ->searchable()
                    ->sortable(),

                ImageColumn::make('thumbnail')
                    ->label('Thumbnail')
                    ->disk('public')
                    ->height(60)
                    ->defaultImageUrl(url('/images/default-video-thumbnail.jpg')),

                IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger')
                    ->sortable(),

                TextColumn::make('video')
                    ->label('Video File')
                    ->limit(30)
                    ->url(fn ($record) => $record->video
                        ? asset('storage/' . ltrim($record->video, '/'))
                        : null, true)
                    ->openUrlInNewTab(),

                TextColumn::make('created_at')
                    ->label('Created At')
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                ViewAction::make(),
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListVideogalleries::route('/'),
            'create' => CreateVideogallery::route('/create'),
            'view' => ViewVideogallery::route('/{record}'),
            'edit' => EditVideogallery::route('/{record}/edit'),
        ];
    }
}
