<?php

namespace App\Filament\Resources\Vehicles;

use App\Filament\Resources\Vehicles\Pages\CreateVehicle;
use App\Filament\Resources\Vehicles\Pages\EditVehicle;
use App\Filament\Resources\Vehicles\Pages\ListVehicles;
use App\Filament\Resources\Vehicles\Pages\ViewVehicle;
use App\Filament\Resources\Vehicles\Schemas\VehicleForm;
use App\Filament\Resources\Vehicles\Schemas\VehicleInfolist;
use App\Filament\Resources\Vehicles\Tables\VehiclesTable;
use App\Models\Vehicle;
use App\Models\Category;
use App\Models\Currency;
use BackedEnum;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Auth;

use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Repeater;

use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\IconColumn;

use Filament\Tables\Filters\SelectFilter;
use Filament\Actions\EditAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;


class VehicleResource extends Resource
{
    protected static ?string $model = Vehicle::class;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedTruck;

    protected static ?string $recordTitleAttribute = 'Vehicle';

    protected static ?int $navigationSort = 1;

    protected static function hasDashboardOption(): bool
    {
        $user = Auth::user();
        if ($user?->role === 'super_admin') {
            return true;
        }
        $options = $user?->dashboard_options ?? [];
        return in_array('vehicles', $options, true);
    }

    public static function canViewAny(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canCreate(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canEdit($record): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canDelete($record): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function shouldRegisterNavigation(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function form(Schema $schema): Schema
    {
          return $schema->components([
         FileUpload::make('image')
             ->disk('public')
             ->directory('vehicles')
             ->image()
             ->maxSize(6144)
             ->visibility('public')
             ->preserveFilenames(false)
             ->required(),

        FileUpload::make('thumbnail')
            ->label('Thumbnail Image')
            ->disk('public')
            ->directory('vehicles/thumbnails')
            ->image()
            ->maxSize(2048)
            ->visibility('public')
            ->preserveFilenames(false)
            ->helperText('Upload a thumbnail image (Max 2MB, recommended: 400x300px)')
            ->nullable(),

        FileUpload::make('sound')
            ->label('Sound (Audio)')
            ->disk('public')
            ->directory('vehicles/sounds')
            ->visibility('public')
            ->maxSize(10240)
            ->nullable(),

        FileUpload::make('schedule_test_drive_pdf')
            ->label('Schedule Test Drive PDF')
            ->acceptedFileTypes(['application/pdf'])
            ->disk('public')
            ->directory('vehicles/documents')
            ->visibility('public')
            ->maxSize(10240)
            ->helperText('Upload the test drive booking form (Max 10MB)')
            ->nullable(),

        FileUpload::make('car_brochure_pdf')
            ->label('Car Brochure PDF')
            ->acceptedFileTypes(['application/pdf'])
            ->disk('public')
            ->directory('vehicles/documents')
            ->visibility('public')
            ->maxSize(10240)
            ->helperText('Upload the vehicle brochure (Max 10MB)')
            ->nullable(),

        \Filament\Forms\Components\TagsInput::make('tags')
            ->label('Tags')
            ->placeholder('Add tags (press Enter after each tag)')
            ->helperText('Add keywords for better search visibility (e.g., luxury, sports, family)')
            ->suggestions([
                'Luxury',
                'Sports',
                'Family',
                'Electric',
                'Hybrid',
                'Compact',
                'SUV',
                'Sedan',
                'Convertible',
                'Off-road',
            ])
            ->columnSpanFull(),

        Repeater::make('featureImages')
            ->label('Feature Images')
            ->relationship()
            ->schema([
                FileUpload::make('image')
                    ->label('Image')
                    ->disk('public')
                    ->directory('vehicles/feature')
                    ->image()
                    ->maxSize(6144)
                    ->visibility('public')
                    ->preserveFilenames(false)
                    ->openable()
                    ->downloadable()
                    ->required(),
            ])
            ->addActionLabel('Add Feature Image')
            ->maxItems(6)
            ->collapsed()
            ->columnSpanFull(),

             Select::make('category_id')
        ->label('Category')
        ->options(Category::all()->pluck('name', 'id'))
        ->searchable()
        ->preload()
        ->required(),

    Toggle::make('is_featured')
        ->label('Featured Vehicle (Premium)')
        ->helperText('Mark as featured to show in premium inventory')
        ->default(false)
        ->inline(false),

    TextInput::make('name')
        ->label('Car Name')
        ->minLength(2)
        ->maxLength(255)
        ->required(),

    TextInput::make('colour')
        ->label('Colour')
        ->maxLength(255)
        ->placeholder('e.g., Black, White, Silver')
        ->nullable(),

    \Filament\Forms\Components\Textarea::make('description')
        ->label('Description')
        ->rows(4)
        ->maxLength(1000)
        ->helperText('Provide a detailed description of the vehicle (max 1000 characters)')
        ->columnSpanFull(),

    TextInput::make('passengers')
        ->label('Passengers')
        ->numeric()
        ->integer()
        ->minValue(1)
        ->required(),

    TextInput::make('km')
        ->label('Mileage (KM)')
        ->numeric()
        ->minValue(0)
        ->required(),

    TextInput::make('cc')
        ->label('Engine CC')
        ->numeric()
        ->integer()
        ->minValue(50)
        ->required(),

    TextInput::make('cylinder')
        ->label('Cylinder')
        ->numeric()
        ->integer()
        ->minValue(1)
        ->maxValue(16)
        ->helperText('Number of cylinders (e.g., 4, 6, 8)')
        ->nullable(),

    TextInput::make('fuel')
        ->label('Fuel Type')
        ->datalist([
            'Petrol',
            'Diesel',
            'Electric',
            'Hybrid',
        ])
        ->required(),

    TextInput::make('price')
        ->label('Price ')
        ->numeric()
        ->step(0.01)
        ->prefix(fn () => Currency::getDefaultSymbol('$'))
        ->required(),

    TextInput::make('discount')
        ->label('Discount (%)')
        ->numeric()
        ->minValue(0)
        ->maxValue(100)
        ->suffix('%')
        ->helperText('Enter discount percentage (0-100)')
        ->nullable(),

        TextInput::make('body_type')
        ->label('Body Type')
        ->datalist([
            'Sedan',
            'Hatchback',
            'SUV',
            'Coupe',
            'Van',
            'Pickup',
        ])
        ->required(),

    TextInput::make('transmission')
        ->label('Transmission')
        ->datalist([
            'Manual',
            'Automatic',
            'Tiptronic',
        ])
        ->required(),

    TextInput::make('drive_type')
        ->label('Drive Type')
        ->datalist([
            'FWD',
            'RWD',
            'AWD',
            '4WD',
        ])
        ->helperText('Front-Wheel Drive (FWD), Rear-Wheel Drive (RWD), All-Wheel Drive (AWD), 4-Wheel Drive (4WD)')
        ->nullable(),

Select::make('make_id')
    ->label('Vehicle Make')
    ->options(\App\Models\Make::pluck('name', 'id'))
    ->searchable()
    ->preload()
    ->nullable(),

    Toggle::make('climate_control')
        ->label('Climate Control')
        ->default(true),

    TextInput::make('mileage')
        ->label('Fuel Mileage (KM/L)')
        ->numeric()
        ->minValue(1)
        ->required(),

    TextInput::make('register_year')
        ->label('Register Year')
        ->numeric()
        ->minValue(1980)
        ->maxValue(now()->year)
        ->required(),

    TextInput::make('condition')
        ->label('Vehicle Condition')
        ->datalist([
            'New',
            'Used',
            'Reconditioned',
        ])
        ->required(),

    TextInput::make('edition')
        ->label('Edition')
        ->placeholder('Optional')
        ->maxLength(255),

    Repeater::make('features')
        ->label('Features / Amenities')
        ->schema([
            TextInput::make('feature')
                ->label('Feature')
                ->required()
                ->maxLength(255),
        ])
        ->addActionLabel('Add Feature')
        ->columnSpanFull(),
]);
    }

    public static function infolist(Schema $schema): Schema
    {
        return VehicleInfolist::configure($schema);
    }

    public static function table(Table $table): Table
    {
        return $table
    ->columns([
        ImageColumn::make('image')
            ->label('Image')
            ->disk('public')
            ->square(),

        TextColumn::make('category.name')
            ->label('Category')
            ->searchable()
            ->sortable(),

        TextColumn::make('name')
            ->label('Car Name')
            ->searchable()
            ->sortable(),

        TextColumn::make('make')
            ->label('Make')
            ->searchable()
            ->sortable(),

        TextColumn::make('body_type')
            ->label('Body Type')
            ->badge()
            ->sortable(),

        TextColumn::make('transmission')
            ->label('Transmission')
            ->badge()
            ->sortable(),

        IconColumn::make('climate_control')
            ->label('Climate')
            ->boolean(),

        TextColumn::make('passengers')
            ->label('Passengers')
            ->sortable(),

        TextColumn::make('km')
            ->label('Mileage (KM)')
            ->sortable(),

        TextColumn::make('mileage')
            ->label('KM / L')
            ->sortable(),

        TextColumn::make('cc')
            ->label('Engine CC')
            ->sortable(),

        TextColumn::make('fuel')
            ->label('Fuel')
            ->badge()
            ->color(fn (string $state): string => match ($state) {
                'Petrol' => 'warning',
                'Diesel' => 'gray',
                'Electric' => 'success',
                'Hybrid' => 'info',
                default => 'secondary',
            }),

        TextColumn::make('register_year')
            ->label('Year')
            ->sortable(),

        TextColumn::make('condition')
            ->label('Condition')
            ->badge()
            ->color(fn (string $state): string => match ($state) {
                'New' => 'success',
                'Used' => 'warning',
                'Reconditioned' => 'info',
                default => 'secondary',
            }),

        TextColumn::make('edition')
            ->label('Edition')
            ->sortable(),

        TextColumn::make('price')
            ->label('Price')
            ->formatStateUsing(fn ($state) => Currency::getDefaultSymbol('Rs.') . (is_numeric($state) ? number_format((float) $state, 2, '.', ',') : (string) $state))
            ->sortable(),

        TextColumn::make('created_at')
            ->label('Created At')
            ->dateTime()
            ->sortable(),
    ])
    ->filters([
        SelectFilter::make('make')
            ->label('Make')
            ->searchable()
            ->preload(),

        SelectFilter::make('body_type')
            ->label('Body Type')
            ->options([
                'Sedan' => 'Sedan',
                'Hatchback' => 'Hatchback',
                'SUV' => 'SUV',
                'Coupe' => 'Coupe',
                'Van' => 'Van',
                'Pickup' => 'Pickup',
            ]),

        SelectFilter::make('transmission')
            ->label('Transmission')
            ->options([
                'Manual' => 'Manual',
                'Automatic' => 'Automatic',
                'Tiptronic' => 'Tiptronic',
            ]),

        SelectFilter::make('fuel')
            ->label('Fuel')
            ->options([
                'Petrol' => 'Petrol',
                'Diesel' => 'Diesel',
                'Electric' => 'Electric',
                'Hybrid' => 'Hybrid',
            ]),

        SelectFilter::make('register_year')
            ->label('Year')
            ->searchable()
            ->preload(),

        SelectFilter::make('condition')
            ->label('Condition')
            ->options([
                'New' => 'New',
                'Used' => 'Used',
                'Reconditioned' => 'Reconditioned',
            ]),

        SelectFilter::make('climate_control')
            ->label('Climate Control')
            ->options([
                1 => 'Yes',
                0 => 'No',
            ]),
    ])
    ->actions([
        EditAction::make(),
        DeleteAction::make(),
    ])
    ->bulkActions([
        BulkActionGroup::make([
            DeleteBulkAction::make(),
        ]),
    ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListVehicles::route('/'),
            'create' => CreateVehicle::route('/create'),
            'view' => ViewVehicle::route('/{record}'),
            'edit' => EditVehicle::route('/{record}/edit'),
        ];
    }
}
