<?php

namespace App\Filament\Resources\Testimonials;

use App\Filament\Resources\Testimonials\Pages;
use App\Models\Testimonial;
use BackedEnum;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Auth;

use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;

use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Actions\EditAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;

class TestimonialsResource extends Resource
{
    protected static ?string $model = Testimonial::class;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedChatBubbleLeftRight;

    protected static ?string $recordTitleAttribute = 'name';

    /* ---------------- Permissions (same logic) ---------------- */

    protected static function hasDashboardOption(): bool
    {
        $user = Auth::user();

        if ($user?->role === 'super_admin') {
            return true;
        }

        $options = $user?->dashboard_options ?? [];
        return in_array('testimonials', $options, true);
    }

    public static function canViewAny(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canCreate(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canEdit($record): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canDelete($record): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function shouldRegisterNavigation(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    /* ---------------- Form ---------------- */

    public static function form(Schema $schema): Schema
    {
        return $schema->components([

            TextInput::make('name')
                ->label('Customer Name')
                ->required()
                ->maxLength(255),

            TextInput::make('designation')
                ->label('Designation')
                ->maxLength(255)
                ->nullable(),

            Select::make('rating')
                ->label('Rating')
                ->options([
                    1 => '⭐',
                    2 => '⭐⭐',
                    3 => '⭐⭐⭐',
                    4 => '⭐⭐⭐⭐',
                    5 => '⭐⭐⭐⭐⭐',
                ])
                ->required(),

            FileUpload::make('image')
                ->label('Profile Image')
                ->disk('public')
                ->directory('testimonials')
                ->image()
                ->visibility('public')
                ->nullable(),

            Textarea::make('review')
                ->label('Review')
                ->rows(5)
                ->columnSpanFull()
                ->required(),

            Toggle::make('status')
                ->label('Active')
                ->default(true),
        ]);
    }

    /* ---------------- Table ---------------- */

    public static function table(Table $table): Table
    {
        return $table
            ->columns([

                ImageColumn::make('image')
                    ->disk('public')
                    ->label('Image')
                    ->circular()
                    ->size(60),

                TextColumn::make('name')
                    ->label('Name')
                    ->searchable()
                    ->sortable(),

                TextColumn::make('designation')
                    ->label('Designation')
                    ->sortable(),

                TextColumn::make('rating')
                    ->label('Rating')
                    ->sortable(),

                ToggleColumn::make('status')
                    ->label('Active'),

                TextColumn::make('created_at')
                    ->label('Created')
                    ->dateTime('d M Y')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->actions([
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTestimonials::route('/'),
            'create' => Pages\CreateTestimonials::route('/create'),
            'view' => Pages\ViewTestimonials::route('/{record}'),
            'edit' => Pages\EditTestimonials::route('/{record}/edit'),
        ];
    }
}
