<?php

namespace App\Filament\Resources\Services;

use App\Filament\Resources\Services\Pages\CreateService;
use App\Filament\Resources\Services\Pages\EditService;
use App\Filament\Resources\Services\Pages\ListServices;
use App\Filament\Resources\Services\Pages\ViewService;
use App\Models\Service;
use BackedEnum;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Auth;

use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Repeater;
use Filament\Tables\Columns\TextColumn;
use Filament\Actions\EditAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;

class ServiceResource extends Resource
{
    protected static ?string $model = Service::class;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedQuestionMarkCircle;

    protected static ?string $navigationLabel = 'FAQ';

    protected static ?string $pluralModelLabel = 'FAQ';

    

    protected static ?string $modelLabel = 'FAQ';

    protected static ?string $recordTitleAttribute = 'FAQ';

    protected static ?int $navigationSort = 2;

    protected static function hasDashboardOption(): bool
    {
        $user = Auth::user();
        if ($user?->role === 'super_admin') {
            return true;
        }
        $options = $user?->dashboard_options ?? [];
        return in_array('services', $options, true);
    }

    public static function canViewAny(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canCreate(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canEdit($record): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canDelete($record): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function shouldRegisterNavigation(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function form(Schema $schema): Schema
    {
        return $schema->components([
            Repeater::make('faqs')
                ->label('FAQ Sections')
                ->schema([
                    TextInput::make('title')
                        ->label('Section Title')
                        ->required()
                        ->maxLength(255),

                    Repeater::make('items')
                        ->label('Questions')
                        ->schema([
                            TextInput::make('question')
                                ->required()
                                ->maxLength(255),

                            Textarea::make('answer')
                                ->required()
                                ->rows(3),
                        ])
                        ->addActionLabel('Add Question')
                        ->collapsed(),
                ])
                ->addActionLabel('Add FAQ Section')
                ->columnSpanFull()
                ->default([
                    ['title' => 'General Information', 'items' => []],
                    ['title' => 'Booking & Reservations', 'items' => []],
                    ['title' => 'Pricing & Payment', 'items' => []],
                    ['title' => 'Vehicle Information', 'items' => []],
                ]),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')->sortable(),

                TextColumn::make('faqs')
                    ->label('Sections Count')
                    ->state(fn ($record) => is_array($record->faqs) ? count($record->faqs) : 0),

                TextColumn::make('created_at')->dateTime()->sortable(),
            ])
            ->actions([
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index'  => ListServices::route('/'),
            'create' => CreateService::route('/create'),
            'view'   => ViewService::route('/{record}'),
            'edit'   => EditService::route('/{record}/edit'),
        ];
    }
}
