<?php

namespace App\Filament\Resources\Currencies;

use App\Filament\Resources\Currencies\Pages\CreateCurrency;
use App\Filament\Resources\Currencies\Pages\EditCurrency;
use App\Filament\Resources\Currencies\Pages\ListCurrencies;
use App\Filament\Resources\Currencies\Pages\ViewCurrency;
use App\Models\Currency;
use BackedEnum;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Auth;

use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;

use Filament\Tables\Columns\TextColumn;
use Filament\Actions\EditAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;

class CurrencyResource extends Resource
{
    protected static ?string $model = Currency::class;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedBanknotes;

    protected static ?string $recordTitleAttribute = 'name';

    protected static ?int $navigationSort = 3;

    protected static function hasDashboardOption(): bool
    {
        $user = Auth::user();

        if ($user?->role === 'super_admin') {
            return true;
        }

        $options = $user?->dashboard_options ?? [];
        return in_array('currencies', $options, true);
    }

    public static function canViewAny(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canCreate(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canEdit($record): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function canDelete($record): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function shouldRegisterNavigation(): bool
    {
        return Auth::check() && static::hasDashboardOption();
    }

    public static function form(Schema $schema): Schema
    {
        return $schema->components([
            TextInput::make('name')
                ->label('Currency Name')
                ->required()
                ->maxLength(255),

            TextInput::make('code')
                ->label('Currency Code')
                ->placeholder('LKR, USD, NZD')
                ->required()
                ->maxLength(10)
                ->unique(ignoreRecord: true),

            TextInput::make('symbol')
                ->label('Symbol')
                ->placeholder('Rs, $, €')
                ->required()
                ->maxLength(10),

            Toggle::make('is_default')
                ->label('Default currency for website')
                ->helperText('If enabled, this symbol will be used on the website.')
                ->default(false),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table->columns([
            TextColumn::make('name')
                ->label('Name')
                ->searchable()
                ->sortable(),

            TextColumn::make('code')
                ->label('Code')
                ->searchable()
                ->sortable(),

            TextColumn::make('symbol')
                ->label('Symbol')
                ->sortable(),

            TextColumn::make('is_default')
                ->label('Default')
                ->formatStateUsing(fn ($state) => $state ? 'Yes' : 'No')
                ->sortable(),

            TextColumn::make('created_at')
                ->label('Created At')
                ->dateTime()
                ->sortable(),
        ])
        ->actions([
            EditAction::make(),
            DeleteAction::make(),
        ])
        ->bulkActions([
            BulkActionGroup::make([
                DeleteBulkAction::make(),
            ]),
        ]);
    }

    public static function getPages(): array
    {
        return [
            'index'  => ListCurrencies::route('/'),
            'create' => CreateCurrency::route('/create'),
            'view'   => ViewCurrency::route('/{record}'),
            'edit'   => EditCurrency::route('/{record}/edit'),
        ];
    }
}
